<?php namespace ProcessWire;

# ---------------------------------------------------------------------
# ImmoImport
# ---------------------------------------------------------------------

### Hilfsfunktionen für Template Code

#######################################################################

# Modul

function immoGetModuleSetting(string $fieldName = '') {

    # ImmoImport Einstellung abfragen
    # Gibt den Feldwert oder undefiniert zurück (Mixed)

    $ImmoImportModule = wire('modules')->isInstalled('ImmoImport') ? wire('modules')->get('ImmoImport') : false;

    if ($fieldName && $ImmoImportModule) {
        return $ImmoImportModule->$fieldName;
    }

    return null;

}

# ---------------------------------------------------------------------

# Objekte

function immoCountProperties(string $template = 'immoimport_expose') {

    # Objekte zählen
    # Gibt die Anzahl 0-X zurück (Integer)

    return wire('pages')->find('template=' . $template)->count;

}

function immoFindProperties(string $filter = '', string $template = 'immoimport_expose') {

    # Objekte finden mit Filter (Selektor)
    # Objekte finden ohne Filter (Selektor)
    # Gibt alle Treffer zurück (ProcessWire PageArray)

    $selector = 'template=' . $template;
    $selector .= $filter ? ', ' . $filter : '';

    return wire('pages')->find($selector);

}

# ---------------------------------------------------------------------

# Objekt

function immoFindTitleImagePage(object $page, string $template = 'immoimport_expose_attachment', string $prefix = 'i_') {

    # Erste Titelbildseite in allen Anhängen (Unterseiten) des Objekts suchen
    # Die Titelbildseite zurückgeben, falls gefunden (Page)
    # Alternativ erste andere Bildseite zurückgeben oder leere Seite (ProcessWire Page oder NullPage)

    if (!($page instanceof Page)) {
        return new NullPage();
    }

    $selector = 'parent=' . $page . ', ';
    $selector .= 'template=' . $template . ', ';
    $selector .= $prefix . 'anh_bild!=0';
    $selectorTitle = $selector . ', ' . $prefix . 'anh_bild_titel=1';
    $titleImagePage = wire('pages')->findOne($selectorTitle);

    if (!$titleImagePage->id) {
        $titleImagePage = wire('pages')->findOne($selector);
    }

    return $titleImagePage;

}

function immoFindImagePages(object $page, string $template = 'immoimport_expose_attachment', string $prefix = 'i_') {

    # Finde alle Bildseiten in allen Anhängen (Unterseiten) des Objekts
    # Alle Bildseiten zurückgeben mit der Titelbildseite an Position eins (ProcessWire PageArray)

    if (!($page instanceof Page)) {
        return new PageArray();
    }

    $selector = 'parent=' . $page . ', ';
    $selector .= 'template=' . $template . ', ';
    $selector .= $prefix . 'anh_bild!=0, ';
    $selector .= 'sort=' . $prefix . 'anh_bild_titel';

    return wire('pages')->find($selector);

}

function immoFindAttachmentPages(object $page, string $template = 'immoimport_expose_attachment', string $prefix = 'i_') {

    # Finde alle Anhänge (Unterseiten) des Objekts
    # Gib alle Anhänge zurück (ProcessWire PageArray)

    if (!($page instanceof Page)) {
        return new PageArray();
    }

    $selector = 'parent=' . $page . ', ';
    $selector .= 'template=' . $template . ', ';
    $selector .= 'sort=sort';

    return wire('pages')->find($selector);

}

# ---------------------------------------------------------------------

# Formatieren

function immoFormatPrice($price = 0, $digits = '', string $symbol = '€') {

    # Preis nach aktiver Spracheinstellung formatieren
    # Erfordert einen definierten PHP setlocale Wert ('de_DE.UTF-8' etc.)
    # Zwei Nachkommastellen sind Standard (gerundet)
    # Gibt Zahl unverändert (Mixed) oder formatiert zurück (String)

    if (!is_numeric($price)) {
        return $price;
    }

    $locale = localeconv();

    if ($locale['frac_digits'] > 3) {
        $locale['frac_digits'] = 2;
    }

    if (!empty($digits) || $digits === 0) {
        if (is_numeric($digits) && $digits >= 0) {
            $locale['frac_digits'] = $digits;
        }
    }

    $price = number_format($price, $locale['frac_digits'], $locale['decimal_point'], $locale['thousands_sep']) . ($symbol ? ' ' . $symbol : '');

    return $price;

}

function immoFormatSpace($number = 0, $digits = '', string $symbol = 'm²') {

    # Platzangabe nach aktiver Spracheinstellung formatieren
    # Erfordert einen definierten PHP setlocale Wert ('de_DE.UTF-8' etc.)
    # Zwei Nachkommastellen sind Standard (gerundet)
    # Gibt Zahl unverändert (Mixed) oder formatiert zurück (String)

    if (!is_numeric($number)) {
        return $number;
    }

    $locale = localeconv();

    if ($locale['frac_digits'] > 3) {
        $locale['frac_digits'] = 2;
    }

    if (!empty($digits) || $digits === 0) {
        if (is_numeric($digits) && $digits >= 0) {
            $locale['frac_digits'] = $digits;
        }
    }

    $number = number_format($number, $locale['frac_digits'], $locale['decimal_point'], $locale['thousands_sep']) . ($symbol ? ' ' . $symbol : '');

    return $number;

}

function immoFormatNumber($number = 0, $digits = '', string $symbol = '') {

    # Zahl nach aktiver Spracheinstellung formatieren
    # Erfordert einen definierten PHP setlocale Wert ('de_DE.UTF-8' etc.)
    # Keine Nachkommastellen sind Standard (gerundet)
    # Gibt Zahl unverändert (Mixed) oder formatiert zurück (String)

    if (!is_numeric($number)) {
        return $number;
    }

    $locale = localeconv();
    $locale['frac_digits'] = 0;

    if (!empty($digits) || $digits === 0) {
        if (is_numeric($digits) && $digits >= 0) {
            $locale['frac_digits'] = $digits;
        }
    }

    $number = number_format($number, $locale['frac_digits'], $locale['decimal_point'], $locale['thousands_sep']) . ($symbol ? ' ' . $symbol : '');

    return $number;

}

# ---------------------------------------------------------------------

# Markup

function immoRenderExposeShort(object $property) {

    # Kurzexposé erzeugen
    # Markup zurückgeben (String)

    $m = '';

    if (!($property instanceof Page)) {
        return $m;
    }

    $m .= '<div style="background-color: #f8f8f8; margin-top: 1em; padding: 0.6em 1.6em">';
    $m .= '    <p><strong>Titel: ' . $property->title . '</strong></p>' . PHP_EOL;

    # Titelbild finden und in einheitlicher Größe ausgeben
    # Alternativer Text falls kein Bild gefunden wurde

    $titleImagePage = immoFindTitleImagePage($property);

    if ($titleImagePage->id) {
        $image = $titleImagePage->i_anh_bild;
        $m .= '    <p><a href="' . $property->url . '"><img alt="Bild" src="' . $image->size(800, 150)->url . '"></a></p>' . PHP_EOL;
    } else {
        $m .= '    <p>Zu diesem Objekt ist leider kein Bild verfügbar.</p>' . PHP_EOL;
    }

    # Objektattribute, Objektart Ort Vermarktungsart

    $objektart = immoGetOptionsString(strtolower($property->i_obj_objektart), [
        'zimmer' => 'Zimmer',
        'wohnung' => 'Wohnung',
        'haus' => 'Haus',
        'grundstueck' => 'Grundstück',
        'buero_praxen' => 'Büro/Praxis',
        'einzelhandel' => 'Einzelhandel',
        'gastgewerbe' => 'Gastgewerbe',
        'hallen_lager_prod' => 'Hallen/Lager/Produktion',
        'land_und_forstwirtschaft' => 'Land- und Forstwirtschaft',
        'parken' => 'Parken',
        'sonstige' => 'Sonstige',
        'freizeitimmobilie_gewerblich' => 'Freizeitimmobilie gewerblich',
        'zinshaus_renditeobjekt' => 'Zinshaus/Renditeobjekt',
        '' => 'Sonstiges Objekt'
    ]);

    if ($property->i_geo_ort) {
        $ort = $property->i_geo_ort;
    } else {
        $ort = 'Unbekannt';
    }

    if ($property->i_obj_vermarktungsart_kauf > 0) {
        $vermarktungsart = ['kauf', 'zum Kauf'];
    } else if ($property->i_obj_vermarktungsart_miete_pacht > 0) {
        $vermarktungsart = ['miet', 'zur Miete/Pacht'];
    } else if ($property->i_obj_vermarktungsart_erbpacht > 0) {
        $vermarktungsart = ['erbp', 'zur Erbpacht'];
    } else if ($property->i_obj_vermarktungsart_leasing > 0) {
        $vermarktungsart = ['leas', 'zum Leasing'];
    }

    $m .= '    <p>' . $objektart . ' in ' . $ort . (!empty($vermarktungsart) ? ' ' . $vermarktungsart[1] : '') . '</p>' . PHP_EOL;

    # Objektattribute, Primäre Fläche abhängig von der Objektart

    $flaeche = immoGetOptionsString(strtolower($property->i_obj_objektart), [
        'zimmer' => ['Wohnfläche', 'i_fla_wohnflaeche'],
        'wohnung' => ['Wohnfläche', 'i_fla_wohnflaeche'],
        'haus' => ['Wohnfläche', 'i_fla_wohnflaeche'],
        'grundstueck' => ['Grundstücksfläche', 'i_fla_grundstuecksflaeche'],
        'buero_praxen' => ['Bürofläche', 'i_fla_bueroflaeche'],
        'einzelhandel' => ['Ladenfläche', 'i_fla_ladenflaeche'],
        'gastgewerbe' => ['Gastrofläche', 'i_fla_gastroflaeche'],
        'hallen_lager_prod' => ['Lagerfläche', 'i_fla_lagerflaeche'],
        'land_und_forstwirtschaft' => ['Grundstücksfläche', 'i_fla_grundstuecksflaeche'],
        'parken' => [],
        'sonstige' => [],
        'freizeitimmobilie_gewerblich' => ['Wohnfläche', 'i_fla_wohnflaeche'],
        'zinshaus_renditeobjekt' => ['Wohnfläche', 'i_fla_wohnflaeche'],
        '' => []
    ]);

    if (!empty($flaeche)) {

        $flaeche_name = $flaeche[0];
        $flaeche_wert = $flaeche[1];
        $flaeche_ausgabe = $flaeche_wert;

        if ($property->hasField($flaeche_wert) && !empty($property->$flaeche_wert)) {
            $flaeche_ausgabe = immoFormatSpace($property->$flaeche_wert);
        } else {
            $flaeche_ausgabe = 'keine Angabe';
        }

        $m .= '    <p>' . $flaeche_name . ': ' . $flaeche_ausgabe . '</p>' . PHP_EOL;

    }

    # Objektattribute, Preisangabe abhängig von der Vermarktungsart

    if (!empty($vermarktungsart)) {

        if ($vermarktungsart[0] === 'kauf') {
            if (!empty($property->i_pre_kaufpreis_auf_anfrage)) {
                $preis = ['Kaufpreis', 'auf Anfrage'];
            } else if (!empty($property->i_pre_kaufpreis)) {
                $preis = ['Kaufpreis', 'i_pre_kaufpreis'];
            }
        } else if ($vermarktungsart[0] === 'miet') {
            if (!empty($property->i_pre_pacht)) {
                $preis = ['Pacht', 'i_pre_pacht'];
            } else if (!empty($property->i_pre_kaltmiete)) {
                $preis = ['Kaltmiete', 'i_pre_kaltmiete'];
            }
        } else if ($vermarktungsart[0] === 'erbp') {
            $preis = ['Erbpacht', 'i_pre_erbpacht'];
        } else if ($vermarktungsart[0] === 'leas') {
            # ?
        }

        if (!empty($preis)) {

            $preis_name = $preis[0];
            $preis_wert = $preis[1];
            $preis_ausgabe = $preis_wert;

            if ($property->hasField($preis_wert) && !empty($property->$preis_wert)) {
                $preis_ausgabe = immoFormatPrice($property->$preis_wert);
            }

            # Bereits vorhandenen Wert wie "auf Anfrage" nicht überschreiben

            if (empty($preis_ausgabe)) {
                $preis_ausgabe = 'keine Angabe';
            }

            $m .= '    <p>' . $preis_name . ': ' . $preis_ausgabe . '</p>' . PHP_EOL;

        }

    }

    $m .= '    <p><a href="' . $property->url . '">Zum Exposé von: ' . $property->title . '</a>' . PHP_EOL;
    $m .= '</div>';

    return $m;

}

function immoRenderAttrList(array $add) {

    # Liste erzeugen
    # Markup zurückgeben (String)

    $m = '';

    foreach ($add as $attr) {
        $m .= '<div style="background-color: #f8f8f8; border-top: 0.1rem solid #343a40;">';
        $m .= '<div style="display: inline-block; vertical-align: top; box-sizing: border-box; width: 30%; padding: 0.5em;">' . $attr['label'] . ':</div>';
        $m .= '<div style="display: inline-block; vertical-align: top; box-sizing: border-box; width: 70%; padding: 0.5em;">' . $attr['value'] . '</div>';
        $m .= '</div>' . PHP_EOL;
    }

    return $m;

}

function immoRenderLinkEmail(string $email = '', array $parameters = []) {

    # Konvertiere E-Mail-Adresse zu ASCII (ohne Unicode in der Domain)
    # Validiere ASCII E-Mail-Adresse und setze bei Erfolg "mailto:" Link
    # Hänge optional weitere Parameter an den "mailto:" Link an (kodiert)
    # Unverändert oder mit Markup zurückgeben (String)

    $email = trim($email);
    $emailInAscii = $email;
    $parametersMarkup = '';
    $i = 0;

    if (function_exists('idn_to_ascii')) {
        if (strpos($email, '@')) {
            list($name, $domain) = explode('@', $email);
            $domain = idn_to_ascii($domain, IDNA_NONTRANSITIONAL_TO_ASCII, INTL_IDNA_VARIANT_UTS46);
            $emailInAscii = $name . '@' . $domain;
        }
    }

    if (filter_var($emailInAscii, FILTER_VALIDATE_EMAIL)) {

        if ($parameters) {
            foreach ($parameters as $name => $value) {
                $parametersMarkup .= $i > 0 ? '&' : '?';
                $parametersMarkup .= rawurlencode($name) . '=' . rawurlencode($value);
                $i++;
            }
        }

        return '<a href="mailto:' . $email . $parametersMarkup . '">' . $email . '</a>';

    }

    return $email;

}

function immoRenderLinkWebsite(string $url = '', array $attributes = []) {

    # Setze Link zur URL und füge optional Attribute hinzu
    # Unverändert oder mit Markup zurückgeben (String)

    $url = trim($url);
    $urlInAscii = $url;
    $attributesMarkup = '';

    if (function_exists('idn_to_ascii')) {
        $urlInAscii = idn_to_ascii($url, IDNA_NONTRANSITIONAL_TO_ASCII, INTL_IDNA_VARIANT_UTS46);
    }

    if (filter_var($urlInAscii, FILTER_VALIDATE_URL)) {

        if ($attributes) {
            foreach ($attributes as $name => $value) {
                $attributesMarkup .= ' ' . $name . '="' . $value . '"';
            }
        }
    
        return '<a href="' . $url . '"' . $attributesMarkup . '>' . $url . '</a>';

    }

    return $url;

}

# ---------------------------------------------------------------------

# Attribute Helfer

function immoGetTableList(string $tableList = '') {

    # Objektattribute aus dem Feld „Tabellenliste“ lesen
    # Konvertierung Text zu Array mit Einzelattributen
    # 'Attribut Name ohne Präfix' => Wert in Typ (Integer usw.)
    # Rückgabe gesammelte Attribute (Array)

    $items = [];

    foreach (explode('[[\]]', $tableList) as $item) {

        $attr = explode('[[|]]', trim($item));

        if (count($attr) < 3) {
            continue;
        }

        $name = $attr[0] ?? '';
        $value = $attr[1] ?? '';
        $type = $attr[2] ?? '';

        switch ($type) {
            case 'decimal': $value = floatval($value); break;
            case 'integer': $value = intval($value); break;
            case 'string': $value = strval($value); break;
            case 'textarea': $value = strval($value); break;
            case 'boolean': if ($value === 'true' || $value === '1') { $value = true; } else { $value = false; }; break;
            case 'datetime': $value = strval($value); break;
            default: strval($value);
        }

        $items[$name] = $value;

    }

    return $items;

}

function immoGetOptionsString(string $fieldname = '', array $options = []) {

    # Vereinfachte Prüfung mit == um verschiedene Typen zu vergleichen
    # Gib transformierten Text zurück falls Übereinstimmung gefunden (String)

    $text = '';

    foreach ($options as $key => $value) {
        if ($fieldname == $key) {
            $text = $value;
            break;
        }
    }

    return $text;

}

function immoFormatAttrList(array $tableList, array $add, string $prefix = '') {

    # Formatiere ein Array mit Attributen (Liste)
    # Setze automatisch Wert und Typ aus „Tabellenliste“ wenn Name identisch
    # Formatiere und oder entferne Attribute je nach Parameter und Kontext
    # Rückgabe der formatierten Liste für die Ausgabe (Array)

    # Definition Ausgabe befüllter und leerer Attributwerte abhängig vom Typ

    $showFilledBooleanAs = 'Ja';

    $showEmptyBooleanAs = 'Nein';
    $showEmptyIntegerAs = 0;
    $showEmptyDoubleAs = 0;
    $showEmptyStringAs = '–';

    # Setze automatisch Wert aus „Tabellenliste“ wenn Name identisch

    foreach ($tableList as $name => $value) {

        foreach ($add as &$attr) {
            if ($prefix . $attr['name'] === $name) {
                $attr['value'] = $value;
            }
        }

        unset($attr);

    }

    # Setze Leerwert falls noch gar kein Wert definiert ist

    foreach ($add as &$attr) {
        if (!isset($attr['value'])) {
            $attr['value'] = '';
        }
    }

    unset($attr);

    # Entferne Einträge ohne Label
    # Entferne Einträge mit Leerwert außer diese bleiben sichtbar
    # Formatiere Feldwert nach Vorgabe
    # Ändere leeren Feldwert abhängig vom Typ und Feldwert
    # Label Anhang anfügen falls definiert und trimmen

    foreach ($add as $key => &$attr) {

        if (empty($attr['label'])) {
            unset($add[$key]);
            continue;
        }

        if (gettype($attr['value']) === 'string') {
            $attr['value'] = trim($attr['value']);
        }

        if (empty($attr['value']) && empty($attr['showEmpty'])) {
            unset($add[$key]);
            continue;
        }

        if (!empty($attr['format'])) {

            $format = !empty($attr['format']) ? $attr['format'] : '';
            $formatDigits = isset($attr['formatDigits']) ? $attr['formatDigits'] : '';
            $formatSymbol = isset($attr['formatSymbol']) ? $attr['formatSymbol'] : '';

            # Parameter Symbol nur überschreiben wenn explizit angefordert

            if ($format === 'space') {
                $attr['value'] = $formatSymbol ? immoFormatSpace($attr['value'], $formatDigits, $formatSymbol) : immoFormatSpace($attr['value'], $formatDigits);
            } else if ($format === 'price') {
                $attr['value'] = $formatSymbol ? immoFormatPrice($attr['value'], $formatDigits, $formatSymbol) : immoFormatPrice($attr['value'], $formatDigits);
            } else if ($format === 'number') {
                $attr['value'] = $formatSymbol ? immoFormatNumber($attr['value'], $formatDigits, $formatSymbol) : immoFormatNumber($attr['value'], $formatDigits);
            }

        }

        $type = gettype($attr['value']);

        if (empty($attr['value'])) {
            if ($type === 'boolean') {
                $attr['value'] = $showEmptyBooleanAs;
            } else if ($type === 'integer') {
                $attr['value'] = $showEmptyIntegerAs;
            } else if ($type === 'double') {
                $attr['value'] = $showEmptyDoubleAs;
            } else if ($type === 'string') {
                $attr['value'] = $showEmptyStringAs;
            }
        } else {
            if ($type === 'boolean') {
                $attr['value'] = $showFilledBooleanAs;
            }
        }

        if (!empty($attr['labelPrepend'])) {
            $attr['value'] = $attr['labelPrepend'] . ' ' . strval($attr['value']);
        }

        if (!empty($attr['labelAppend'])) {
            $attr['value'] = strval($attr['value']) . ' ' . $attr['labelAppend'];
        }

        if (gettype($attr['value']) === 'string') {
            $attr['value'] = trim($attr['value']);
        }

    }

    unset($attr);

    return $add;

}

# ---------------------------------------------------------------------

# Attribute Listen (Beispiele Aufbau eigener Listen)

function immoAttrListExample(array $tableList, object $page) {

    # Beispiel Definition Liste mit beliebigen Objektattributen
    # Attribute formatieren, kombinieren, beschreiben, neu definieren
    # Leere Werte werden standardmäßig nicht ausgegeben (0 oder '')
    # Markup der fertigen Liste wird zurückgeben (String)

    # 'name' => 'String' (Pflicht)
    # 'label' => 'String' (Pflicht)
    # 'value' => Mixed
    # 'showEmpty' => Boolean true|false
    # 'format' => 'String price|space|number'
    # 'formatDigits' => Integer 0-X
    # 'formatSymbol' => 'String'
    # 'labelPrepend' => 'String'
    # 'labelAppend' => 'String'

    $add = [];

    # Bei Leerwert erfolgt keine Ausgabe des Attributs (wenn 0 oder '')
    # Ausgabe eines Leerwerts kann mit 'showEmpty' erzwungen werden

    $add[] = ['name' => 'geo_ort', 'label' => 'Ort'];
    $add[] = ['name' => 'custom_foo', 'label' => 'Foo (Text auch leer sichtbar)', 'value' => '', 'showEmpty' => true, 'labelAppend' => '(Hinweis)'];
    $add[] = ['name' => 'custom_bar', 'label' => 'Bar (Boolean auch leer sichtbar)', 'value' => false, 'showEmpty' => true];

    # Formatierung Preis in verschiedenen Varianten

    $add[] = ['name' => 'pre_kaufpreis', 'label' => 'Kaufpreis 1 (formatiert Standard)', 'format' => 'price'];
    $add[] = ['name' => 'pre_kaufpreis', 'label' => 'Kaufpreis 2 (formatiert Option)', 'format' => 'price', 'formatDigits' => 0];
    $add[] = ['name' => 'pre_kaufpreis', 'label' => 'Kaufpreis 3 (formatiert Option)', 'format' => 'price', 'formatSymbol' => 'Euro'];
    $add[] = ['name' => 'pre_kaufpreis', 'label' => 'Kaufpreis 4 (formatiert Option)', 'format' => 'price', 'formatDigits' => 1, 'formatSymbol' => 'Taler'];

    # Formatierung Platz in verschiedenen Varianten

    $add[] = ['name' => 'fla_wohnflaeche', 'label' => 'Wohnfläche 1 (formatiert Standard)', 'format' => 'space'];
    $add[] = ['name' => 'fla_wohnflaeche', 'label' => 'Wohnfläche 2 (formatiert Option)', 'format' => 'space', 'formatDigits' => 0];
    $add[] = ['name' => 'fla_wohnflaeche', 'label' => 'Wohnfläche 3 (formatiert Option)', 'format' => 'space', 'formatSymbol' => 'Quadratmeter'];
    $add[] = ['name' => 'fla_wohnflaeche', 'label' => 'Wohnfläche 4 (formatiert Option)', 'format' => 'space', 'formatDigits' => 3, 'formatSymbol' => 'm hoch 2'];

    # Formatierung Nummer in verschiedenen Varianten

    $add[] = ['name' => 'geo_anzahl_etagen', 'label' => 'Anzahl Etagen 1 (formatiert Standard)', 'format' => 'number'];
    $add[] = ['name' => 'geo_anzahl_etagen', 'label' => 'Anzahl Etagen 2 (formatiert Option)', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'geo_anzahl_etagen', 'label' => 'Anzahl Etagen 3 (formatiert Option)', 'format' => 'number', 'formatSymbol' => 'Etagen'];
    $add[] = ['name' => 'geo_anzahl_etagen', 'label' => 'Anzahl Etagen 4 (formatiert Option)', 'format' => 'number', 'formatDigits' => 2, 'formatSymbol' => 'Etagen'];

    # Attribute neu definieren und frei befüllen im Kontext von irgendetwas

    if (empty($tableList['veo_objektadresse_freigeben'])) {
        $custom_strasse_hausnummer_ort = 'auf Anfrage (nicht öffentlich)';
    } else {
        $custom_strasse_hausnummer_ort = $tableList['geo_strasse'] . ' ' . $tableList['geo_hausnummer'];
    }

    $add[] = ['name' => 'custom_strasse_hausnummer_ort', 'label' => 'Adresse', 'value' => $custom_strasse_hausnummer_ort];
    $add[] = ['name' => 'custom_datum_heute', 'label' => 'Datum heute', 'value' => date("d.m.Y", time())];
    $add[] = ['name' => 'custom_wohn_und_grundstuecksflaeche', 'label' => 'Summe der Wohn- und Grundstücksfläche', 'value' => $tableList['fla_wohnflaeche'] + $tableList['fla_grundstuecksflaeche'], 'showEmpty' => true, 'format' => 'space'];
    $add[] = ['name' => 'custom_summe_attribute', 'label' => 'Summe der verfügbaren Objektattribute', 'value' => count($tableList)];
    $add[] = ['name' => 'custom_summe_anhaenge', 'label' => 'Summe aller Anhänge', 'value' => $page->children->count, 'format' => 'number', 'formatSymbol' => 'Anhänge total'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListEnergiepass(array $tableList, object $page) {

    # Beispiel Definition Liste Energiepass
    # Attribute neu definieren und befüllen abhängig von Rohdaten

    $energieausweis = immoGetOptionsString(strtoupper($tableList['zus_energiepass_epart']), [
        'BEDARF' => 'Bedarf',
        'VERBRAUCH' => 'Verbrauch'
    ]);

    $energieverbrauchswert = '';
    $endenergiebedarf = '';

    if (strtoupper($tableList['zus_energiepass_epart']) === 'VERBRAUCH') {
        $energieverbrauchswert .= !empty($tableList['zus_energiepass_energieverbrauchkennwert']) ? immoFormatNumber($tableList['zus_energiepass_energieverbrauchkennwert'], 2) . ' kWh/(m²*a)' : '';
        $energieverbrauchswert .= !empty($tableList['zus_energiepass_energieverbrauchkennwert']) && !empty($tableList['zus_energiepass_mitwarmwasser']) ? ' inkl. Warmwasser' : '';
    } else {
        $endenergiebedarf .= !empty($tableList['zus_energiepass_endenergiebedarf']) ? immoFormatNumber($tableList['zus_energiepass_endenergiebedarf'], 2) . ' kWh/(m²*a)' : '';
        $endenergiebedarf .= !empty($tableList['zus_energiepass_endenergiebedarf']) && !empty($tableList['zus_energiepass_mitwarmwasser']) ? ' inkl. Warmwasser' : '';
    }

    $austelldatum = !empty($tableList['zus_energiepass_ausstelldatum']) ? date('d.m.Y', strtotime($tableList['zus_energiepass_ausstelldatum'])) : '';
    $gueltig_bis = !empty($tableList['zus_energiepass_gueltig_bis']) ? $tableList['zus_energiepass_gueltig_bis'] : '';
    $stromwert = !empty($tableList['zus_energiepass_stromwert']) ? immoFormatNumber($tableList['zus_energiepass_stromwert'], 2) . ' kWh/(m²*a)' : '';
    $waermewert = !empty($tableList['zus_energiepass_waermewert']) ? immoFormatNumber($tableList['zus_energiepass_waermewert'], 2) . ' kWh/(m²*a)' : '';
    $wertklasse = !empty($tableList['zus_energiepass_wertklasse']) ? $tableList['zus_energiepass_wertklasse'] : '';
    $baujahr = !empty($tableList['zus_energiepass_baujahr']) ? $tableList['zus_energiepass_baujahr'] : '';
    $geg2018 = !empty($tableList['zus_energiepass_geg2018']) ? $tableList['zus_energiepass_geg2018'] : '';

    $primaerenergietraeger = immoGetOptionsString(strtoupper($tableList['zus_energiepass_primaerenergietraeger']), [
        'OEL' => 'Öl',
        'GAS' => 'Gas',
        'ELEKTRO' => 'Elektro',
        'ALTERNATIV' => 'Alternativ',
        'SOLAR' => 'Solar',
        'ERDWAERME' => 'Wärmepumpe Sole-Wasser',
        'LUFTWP' => 'Wärmepumpe Luft-Wasser',
        'FERN' => 'Fernwärme',
        'BLOCK' => 'Blockheizkraftwerk',
        'WASSER-ELEKTRO' => 'Ergänzendes dezentrales Warmwasser',
        'PELLET' => 'Pellets',
        'KOHLE' => 'Kohle',
        'HOLZ' => 'Holz und Hackschnitzel',
        'FLUESSIGGAS' => 'Flüssiggas'
    ]);

    $jahrgang = immoGetOptionsString(strtolower($tableList['zus_energiepass_jahrgang']), [
        '2008' => '2008',
        '2014' => '2014 ',
        'ohne' => 'ohne',
        'nicht_noetig' => 'nicht nötig',
        'bei_besichtigung' => 'bei Besichtigung'
    ]);

    $gebaeudeart = immoGetOptionsString(strtolower($tableList['zus_energiepass_gebaeudeart']), [
        'wohn' => 'Wohngebäude',
        'nichtwohn' => 'Nichtwohngebäude'
    ]);

    $heizungsart = '';
    $heizungsart .= $tableList['aus_heizungsart_ofen'] ? 'Ofen, ' : '';
    $heizungsart .= $tableList['aus_heizungsart_etage'] ? 'Etage, ' : '';
    $heizungsart .= $tableList['aus_heizungsart_zentral'] ? 'zentral, ' : '';
    $heizungsart .= $tableList['aus_heizungsart_fern'] ? 'fern, ' : '';
    $heizungsart .= $tableList['aus_heizungsart_fussboden'] ? 'Fußboden, ' : '';

    if (substr($heizungsart, -strlen(', ')) === ', ') {
        $heizungsart = substr($heizungsart, 0, -strlen(', '));
    }

    $add = [];

    $add[] = ['name' => 'custom_epass_baujahr', 'label' => 'Baujahr', 'value' => $baujahr];
    $add[] = ['name' => 'custom_epass_gebaeudeart', 'label' => 'Gebäudeart', 'value' => $gebaeudeart];
    $add[] = ['name' => 'custom_epass_austelldatum', 'label' => 'Austelldatum', 'value' => $austelldatum];
    $add[] = ['name' => 'custom_epass_gueltig_bis', 'label' => 'Gültig bis', 'value' => $gueltig_bis];
    $add[] = ['name' => 'custom_epass_energieausweis', 'label' => 'Energieausweis', 'value' => $energieausweis];
    $add[] = ['name' => 'custom_epass_energieverbrauchswert', 'label' => 'Energieverbrauchswert', 'value' => $energieverbrauchswert];
    $add[] = ['name' => 'custom_epass_endenergiebedarf', 'label' => 'Endenergiebedarf', 'value' => $endenergiebedarf];
    $add[] = ['name' => 'custom_epass_stromwert', 'label' => 'Stromwert', 'value' => $stromwert];
    $add[] = ['name' => 'custom_epass_waermewert', 'label' => 'Wärmewert', 'value' => $waermewert];
    $add[] = ['name' => 'custom_epass_wertklasse', 'label' => 'Energieeffizienklasse', 'value' => $wertklasse];
    $add[] = ['name' => 'custom_epass_heizungsart', 'label' => 'Heizungsart', 'value' => $heizungsart];
    $add[] = ['name' => 'custom_epass_primaerenergietraeger', 'label' => 'Wesentlicher Energieträger', 'value' => $primaerenergietraeger];
    $add[] = ['name' => 'custom_epass_jahrgang', 'label' => 'Jahrgang Energieausweis', 'value' => $jahrgang];
    $add[] = ['name' => 'custom_epass_geg2018', 'label' => 'Gebäude Energie Gesetz 2018', 'value' => $geg2018];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

# ---------------------------------------------------------------------

# Attribute Listen (Basisdaten)

function immoAttrListAnbBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'anb_anbieternr', 'label' => 'Anbieter Nummer'];
    # $add[] = ['name' => 'anb_firma', 'label' => 'Firma'];
    # $add[] = ['name' => 'anb_openimmo_anid', 'label' => 'OpenImmo Anbieter ID'];
    # $add[] = ['name' => 'anb_lizenzkennung', 'label' => 'Lizenzkennung'];
    # $add[] = ['name' => 'anb_impressum', 'label' => 'Impressum'];
    $add[] = ['name' => 'anb_impressum_strukt_firmenname', 'label' => 'Anbieter'];
    $add[] = ['name' => 'anb_impressum_strukt_firmenanschrift', 'label' => 'Adresse'];
    $add[] = ['name' => 'anb_impressum_strukt_telefon', 'label' => 'Telefon'];
    # $add[] = ['name' => 'anb_impressum_strukt_vertretungsberechtigter', 'label' => 'Impressum Strukt Vertretungsberechtigter'];
    # $add[] = ['name' => 'anb_impressum_strukt_berufsaufsichtsbehoerde', 'label' => 'Impressum Strukt Berufsaufsichtsbehörde'];
    # $add[] = ['name' => 'anb_impressum_strukt_handelsregister', 'label' => 'Impressum Strukt Handelsregister'];
    # $add[] = ['name' => 'anb_impressum_strukt_handelsregister_nr', 'label' => 'Impressum Strukt Handelsregister Nummer'];
    # $add[] = ['name' => 'anb_impressum_strukt_umsst_id', 'label' => 'Impressum Strukt USt ID'];
    # $add[] = ['name' => 'anb_impressum_strukt_steuernummer', 'label' => 'Impressum Strukt Steuernummer'];
    # $add[] = ['name' => 'anb_impressum_strukt_weiteres', 'label' => 'Impressum Strukt Weiteres'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListObjBasic(array $tableList) {

    $custom_obj_objektart = '';
    $custom_obj_objektart_typ = '';

    $custom_obj_objektart = immoGetOptionsString(strtolower($tableList['obj_objektart']), [
        'zimmer' => 'Zimmer',
        'wohnung' => 'Wohnung',
        'haus' => 'Haus',
        'grundstueck' => 'Grundstück',
        'buero_praxen' => 'Büro/Praxis',
        'einzelhandel' => 'Einzelhandel',
        'gastgewerbe' => 'Gastgewerbe',
        'hallen_lager_prod' => 'Hallen/Lager/Produktion',
        'land_und_forstwirtschaft' => 'Land- und Forstwirtschaft',
        'parken' => 'Parken',
        'sonstige' => 'Sonstige',
        'freizeitimmobilie_gewerblich' => 'Freizeitimmobilie gewerblich',
        'zinshaus_renditeobjekt' => 'Zinshaus/Renditeobjekt',
        '' => 'Sonstige'
    ]);

    # Auslassen da kein Mehrwert (Art: Zimmer, Typ: Zimmer)

    # if (strtolower($tableList['obj_objektart']) === 'zimmer') {
    #     $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
    #         'ZIMMER' => 'Zimmer'
    #     ]);
    # }

    if (strtolower($tableList['obj_objektart']) === 'wohnung') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'DACHGESCHOSS' => 'Dachgeschoss',
            'MAISONETTE' => 'Maisonette',
            'LOFT-STUDIO-ATELIER' => 'Loft/Studio/Atelier',
            'PENTHOUSE' => 'Penthouse',
            'TERRASSEN' => 'Terrassen',
            'ETAGE' => 'Etage',
            'ERDGESCHOSS' => 'Erdgeschoss',
            'SOUTERRAIN' => 'Souterrain',
            'APARTMENT' => 'Apartment',
            'FERIENWOHNUNG' => 'Ferienwohnung',
            'GALERIE' => 'Galerie',
            'ROHDACHBODEN' => 'Rohdachboden',
            'ATTIKAWOHNUNG' => 'Attika',
            'KEINE_ANGABE' => 'Keine Angabe'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'haus') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'REIHENHAUS' => 'Reihenhaus',
            'REIHENEND' => 'Reihenendhaus',
            'REIHENMITTEL' => 'Reihenmittelhaus',
            'REIHENECK' => 'Reiheneckhaus',
            'DOPPELHAUSHAELFTE' => 'Doppelhaushälfte',
            'EINFAMILIENHAUS' => 'Einfamilienhaus',
            'STADTHAUS' => 'Stadthaus',
            'BUNGALOW' => 'Bungalow',
            'VILLA' => 'Villa',
            'RESTHOF' => 'Resthof',
            'BAUERNHAUS' => 'Bauernhaus',
            'LANDHAUS' => 'Landhaus',
            'SCHLOSS' => 'Schloss',
            'ZWEIFAMILIENHAUS' => 'Zweifamilienhaus',
            'MEHRFAMILIENHAUS' => 'Mehrfamilienhaus',
            'FERIENHAUS' => 'Ferienhaus',
            'BERGHUETTE' => 'Berghütte',
            'CHALET' => 'Chalet',
            'STRANDHAUS' => 'Strandhaus',
            'LAUBE-DATSCHE-GARTENHAUS' => 'Laube/Datsche/Gartenhaus',
            'APARTMENTHAUS' => 'Apartmenthaus',
            'BURG' => 'Burg',
            'HERRENHAUS' => 'Herrenhaus',
            'FINCA' => 'Finca',
            'RUSTICO' => 'Rustico',
            'FERTIGHAUS' => 'Fertighaus',
            'KEINE_ANGABE' => 'Keine Angabe'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'grundstueck') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'WOHNEN' => 'Wohnen',
            'GEWERBE' => 'Gewerbe',
            'INDUSTRIE' => 'Industrie',
            'LAND_FORSTWIRSCHAFT' => 'Land und Forstwirschaft',
            'FREIZEIT' => 'Freizeit',
            'GEMISCHT' => 'Gemischt',
            'GEWERBEPARK' => 'Gewerbepark',
            'SONDERNUTZUNG' => 'Sondernutzung',
            'SEELIEGENSCHAFT' => 'Seeliegenschaft'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'buero_praxen') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'BUEROFLAECHE' => 'Bürofläche',
            'BUEROHAUS' => 'Bürohaus',
            'BUEROZENTRUM' => 'Bürozentrum',
            'LOFT_ATELIER' => 'Loft-Atelier',
            'PRAXIS' => 'Praxis',
            'PRAXISFLAECHE' => 'Praxisfläche',
            'PRAXISHAUS' => 'Praxishaus',
            'AUSSTELLUNGSFLAECHE' => 'Ausstellungsfläche',
            'COWORKING' => 'Coworking',
            'SHARED_OFFICE' => 'Shared Office'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'einzelhandel') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'LADENLOKAL' => 'Ladenlokal',
            'EINZELHANDELSLADEN' => 'Einzelhandelsladen',
            'VERBRAUCHERMARKT' => 'Verbrauchermarkt',
            'EINKAUFSZENTRUM' => 'Einkaufszentrum',
            'KAUFHAUS' => 'Kaufhaus',
            'FACTORY_OUTLET' => 'Factory Outlet',
            'KIOSK' => 'Kiosk',
            'VERKAUFSFLAECHE' => 'Verkaufsfläche',
            'AUSSTELLUNGSFLAECHE' => 'Ausstellungsfläche'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'gastgewerbe') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'GASTRONOMIE' => 'Gastronomie',
            'GASTRONOMIE_UND_WOHNUNG' => 'Gastronomie und Wohnung',
            'PENSIONEN' => 'Pensionen',
            'HOTELS' => 'Hotels',
            'WEITERE_BEHERBERGUNGSBETRIEBE' => 'Weitere Beherbergungsbetriebe',
            'BAR' => 'Bar',
            'CAFE' => 'Café',
            'DISCOTHEK' => 'Discothek',
            'RESTAURANT' => 'Restaurant',
            'RAUCHERLOKAL' => 'Raucherlokal',
            'EINRAUMLOKAL' => 'Einraumlokal'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'hallen_lager_prod') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'HALLE' => 'Halle',
            'INDUSTRIEHALLE' => 'Industriehalle',
            'LAGER' => 'Lager',
            'LAGERFLAECHEN' => 'Lagerflächen',
            'LAGER_MIT_FREIFLAECHE' => 'Lager mit Freifläche',
            'HOCHREGALLAGER' => 'Hochregallager',
            'SPEDITIONSLAGER' => 'Speditionslager',
            'PRODUKTION' => 'Produktion',
            'WERKSTATT' => 'Werkstatt',
            'SERVICE' => 'Service',
            'FREIFLAECHEN' => 'Freiflächen',
            'KUEHLHAUS' => 'Kühlhaus'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'land_und_forstwirtschaft') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'LANDWIRTSCHAFTLICHE_BETRIEBE' => 'Landwirtschaftliche Betriebe',
            'BAUERNHOF' => 'Bauernhof',
            'AUSSIEDLERHOF' => 'Aussiedlerhof',
            'GARTENBAU' => 'Gartenbau',
            'ACKERBAU' => 'Ackerbau',
            'WEINBAU' => 'Weinbau',
            'VIEHWIRTSCHAFT' => 'Viehwirtschaft',
            'JAGD_UND_FORSTWIRTSCHAFT' => 'Jagd und Forstwirtschaft',
            'TEICH_UND_FISCHWIRTSCHAFT' => 'Teich und Fischwirtschaft',
            'SCHEUNEN' => 'Scheunen',
            'REITERHOEFE' => 'Reiterhöfe',
            'SONSTIGE_LANDWIRTSCHAFTSIMMOBILIEN' => 'Sonstige Landwirtschaftsimmobilien',
            'ANWESEN' => 'Anwesen',
            'JAGDREVIER' => 'Jagdrevier'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'parken') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'STELLPLATZ' => 'Stellplatz',
            'CARPORT' => 'Carport',
            'DOPPELGARAGE' => 'Doppelgarage',
            'DUPLEX' => 'Duplex',
            'TIEFGARAGE' => 'Tiefgarage',
            'BOOTSLIEGEPLATZ' => 'Bootsliegeplatz',
            'EINZELGARAGE' => 'Einzelgarage',
            'PARKHAUS' => 'Parkhaus',
            'TIEFGARAGENSTELLPLATZ' => 'Tiefgaragenstellplatz',
            'PARKPLATZ_STROM' => 'Parkplatz mit Ladestation (Strom)'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'sonstige') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'PARKHAUS' => 'Parkhaus',
            'TANKSTELLE' => 'Tankstelle',
            'KRANKENHAUS' => 'Krankenhaus',
            'SONSTIGE' => 'Sonstige'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'freizeitimmobilie_gewerblich') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'SPORTANLAGEN' => 'Sportanlagen',
            'VERGNUEGUNGSPARKS_UND_CENTER' => 'Vergnügungsparks und Center',
            'FREIZEITANLAGE' => 'Freizeitanlage'
        ]);
    }

    if (strtolower($tableList['obj_objektart']) === 'zinshaus_renditeobjekt') {
        $custom_obj_objektart_typ = immoGetOptionsString(strtoupper($tableList['obj_objektart_typ']), [
            'MEHRFAMILIENHAUS' => 'Mehrfamilienhaus',
            'WOHN_UND_GESCHAEFTSHAUS' => 'Wohn und Geschäftshaus',
            'GESCHAEFTSHAUS' => 'Geschäftshaus',
            'BUEROGEBAEUDE' => 'Bürogebäude',
            'SB_MAERKTE' => 'SB-Märkte',
            'EINKAUFSCENTREN' => 'Einkaufscentren',
            'WOHNANLAGEN' => 'Wohnanlagen',
            'VERBRAUCHERMAERKTE' => 'Verbrauchermaerkte',
            'INDUSTRIEANLAGEN' => 'Industrieanlagen',
            'PFLEGEHEIM' => 'Pflegeheim',
            'SANATORIUM' => 'Sanatorium',
            'SENIORENHEIM' => 'Seniorenheim',
            'BETREUTES-WOHNEN' => 'Betreutes Wohnen'
        ]);
    }

    $add = [];

    $add[] = ['name' => 'obj_nutzungsart_wohnen', 'label' => 'Wohnobjekt'];
    $add[] = ['name' => 'obj_nutzungsart_gewerbe', 'label' => 'Gewerbeobjekt'];
    $add[] = ['name' => 'obj_nutzungsart_anlage', 'label' => 'Anlageobjekt'];
    $add[] = ['name' => 'obj_nutzungsart_waz', 'label' => 'WAZ-Objekt (Wohnen auf Zeit)'];
    $add[] = ['name' => 'obj_vermarktungsart_kauf', 'label' => 'Zum Kauf'];
    $add[] = ['name' => 'obj_vermarktungsart_miete_pacht', 'label' => 'Zur Miete/Pacht'];
    $add[] = ['name' => 'obj_vermarktungsart_erbpacht', 'label' => 'Zur Erbpacht'];
    $add[] = ['name' => 'obj_vermarktungsart_leasing', 'label' => 'Zum Leasing'];
    $add[] = ['name' => 'custom_obj_objektart', 'label' => 'Objektart', 'value' => $custom_obj_objektart];
    $add[] = ['name' => 'custom_obj_objektart_typ', 'label' => 'Typ', 'value' => $custom_obj_objektart_typ];
    # $add[] = ['name' => 'obj_objektart_objektart_zusatz', 'label' => 'Zusatz'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListGeoBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'custom_geo_plz_ort', 'label' => 'PLZ und Ort', 'value' => $tableList['geo_plz'] . ' ' . $tableList['geo_ort']];
    # $add[] = ['name' => 'geo_geokoordinaten_breitengrad', 'label' => 'Geokoordinaten Breitengrad'];
    # $add[] = ['name' => 'geo_geokoordinaten_laengengrad', 'label' => 'Geokoordinaten Längengrad'];

    if (empty($tableList['veo_objektadresse_freigeben'])) {
        $custom_geo_strasse_hausnummer = 'auf Anfrage';
    } else {
        $custom_geo_strasse_hausnummer = $tableList['geo_strasse'] . ' ' . $tableList['geo_hausnummer'];
    }

    $add[] = ['name' => 'custom_geo_strasse_hausnummer', 'label' => 'Adresse', 'value' => $custom_geo_strasse_hausnummer];
    # $add[] = ['name' => 'geo_bundesland', 'label' => 'Bundesland'];
    # $add[] = ['name' => 'geo_land_iso_land', 'label' => 'Land ISO Land'];
    # $add[] = ['name' => 'geo_gemeindecode', 'label' => 'Gemeindecode'];
    $add[] = ['name' => 'geo_flur', 'label' => 'Flurangabe'];
    # $add[] = ['name' => 'geo_flurstueck', 'label' => 'Flurstück'];
    # $add[] = ['name' => 'geo_gemarkung', 'label' => 'Gemarkung'];
    $add[] = ['name' => 'geo_etage', 'label' => 'Etage', 'format' => 'number'];
    $add[] = ['name' => 'geo_anzahl_etagen', 'label' => 'Etagen Anzahl', 'format' => 'number'];

    $custom_geo_lage_im_bau = '';
    $custom_geo_lage_im_bau .= $tableList['geo_lage_im_bau_links'] ? 'links ' : '';
    $custom_geo_lage_im_bau .= $tableList['geo_lage_im_bau_rechts'] ? 'rechts ' : '';
    $custom_geo_lage_im_bau .= $tableList['geo_lage_im_bau_vorne'] ? 'vorne ' : '';
    $custom_geo_lage_im_bau .= $tableList['geo_lage_im_bau_hinten'] ? 'hinten ' : '';
    $custom_geo_lage_im_bau = trim($custom_geo_lage_im_bau);

    $add[] = ['name' => 'custom_geo_lage_im_bau', 'label' => 'Lage im Bau', 'value' => $custom_geo_lage_im_bau];
    # $add[] = ['name' => 'geo_wohnungsnr', 'label' => 'Wohnungsnummer'];

    $custom_geo_lage_gebiet_gebiete = immoGetOptionsString(strtoupper($tableList['geo_lage_gebiet_gebiete']), [
        'WOHN' => 'Wohngebiet',
        'GEWERBE' => 'Gewerbegebiet',
        'INDUSTRIE' => 'Industriegebiet',
        'MISCH' => 'Mischgebiet',
        'NEUBAU' => 'Neubaugebiet',
        'ORTSLAGE' => 'Ortslage',
        'SIEDLUNG' => 'Siedlung',
        'STADTRAND' => 'Stadtrand',
        'STADTTEIL' => 'Stadtteil',
        'STADTZENTRUM' => 'Stadtzentrum',
        'NEBENZENTRUM' => 'Nebenzentrum',
        '1A' => '1A',
        '1B' => '1B'
    ]);

    $add[] = ['name' => 'custom_geo_lage_gebiet_gebiete', 'label' => 'Gebietslage', 'value' => $custom_geo_lage_gebiet_gebiete];
    $add[] = ['name' => 'geo_regionaler_zusatz', 'label' => 'Regionaler Zusatz'];
    # $add[] = ['name' => 'geo_karten_makro', 'label' => 'Anzeige Karte Makro'];
    # $add[] = ['name' => 'geo_karten_mikro', 'label' => 'Anzeige Karte Mikro'];
    # $add[] = ['name' => 'geo_virtuelletour', 'label' => 'Objekt hat virtuelle Tour'];
    # $add[] = ['name' => 'geo_luftbildern', 'label' => 'Objekt hat Luftbilder'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListKonBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'custom_kon_vorname_name_anrede', 'label' => 'Name', 'value' => $tableList['kon_anrede'] . ' ' . $tableList['kon_vorname'] . ' ' . $tableList['kon_name']];
    # $add[] = ['name' => 'kon_titel', 'label' => 'Titel'];
    # $add[] = ['name' => 'kon_anrede_brief', 'label' => 'Anrede Brief'];
    # $add[] = ['name' => 'kon_firma', 'label' => 'Firma'];
    # $add[] = ['name' => 'kon_zusatzfeld', 'label' => 'Zusatzfeld'];
    # $add[] = ['name' => 'kon_strasse', 'label' => 'Straße'];
    # $add[] = ['name' => 'kon_hausnummer', 'label' => 'Hausnummer'];
    # $add[] = ['name' => 'kon_plz', 'label' => 'PLZ'];
    # $add[] = ['name' => 'kon_ort', 'label' => 'Ort'];
    # $add[] = ['name' => 'kon_postfach', 'label' => 'Postfach'];
    # $add[] = ['name' => 'kon_postf_plz', 'label' => 'Postfach PLZ'];
    # $add[] = ['name' => 'kon_postf_ort', 'label' => 'Postfach Ort'];
    # $add[] = ['name' => 'kon_land_iso_land', 'label' => 'Land ISO Land'];
    # $add[] = ['name' => 'kon_email_zentrale', 'label' => 'E-Mail Zentrale'];
    $add[] = ['name' => 'custom_kon_email_direkt', 'label' => 'E-Mail-Adresse', 'value' => immoRenderLinkEmail($tableList['kon_email_direkt'], ['subject' => 'Betrifft Objekt ' . $tableList['vet_objektnr_extern']])];
    # $add[] = ['name' => 'kon_email_privat', 'label' => 'E-Mail privat'];
    # $add[] = ['name' => 'kon_email_sonstige', 'label' => 'E-Mail sonstige'];
    # $add[] = ['name' => 'kon_email_sonstige_emailart', 'label' => 'E-Mail sonstige Art'];
    # $add[] = ['name' => 'kon_email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung'];
    $add[] = ['name' => 'kon_tel_durchw', 'label' => 'Telefon (Durchwahl)'];
    # $add[] = ['name' => 'kon_tel_zentrale', 'label' => 'Telefon Zentrale'];
    $add[] = ['name' => 'kon_tel_handy', 'label' => 'Telefon (mobil)'];
    # $add[] = ['name' => 'kon_tel_fax', 'label' => 'Telefon Fax'];
    # $add[] = ['name' => 'kon_tel_privat', 'label' => 'Telefon privat'];
    # $add[] = ['name' => 'kon_tel_sonstige', 'label' => 'Telefon sonstige'];
    # $add[] = ['name' => 'kon_tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart'];
    # $add[] = ['name' => 'kon_tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung'];
    $add[] = ['name' => 'custom_kon_url', 'label' => 'Website', 'value' => immoRenderLinkWebsite($tableList['kon_url'], ['target' => '_blank', 'rel' => 'nofollow'])];
    # $add[] = ['name' => 'kon_adressfreigabe', 'label' => 'Adressfreigabe'];
    # $add[] = ['name' => 'kon_personennummer', 'label' => 'Personennummer'];
    # $add[] = ['name' => 'kon_freitextfeld', 'label' => 'Freitextfeld'];
    # $add[] = ['name' => 'kon_position', 'label' => 'Position'];
    # $add[] = ['name' => 'kon_email_feedback', 'label' => 'E-Mail Feedback'];
    # $add[] = ['name' => 'kon_immobilientreuhaenderid', 'label' => 'Immobilientreuhänder ID'];
    # $add[] = ['name' => 'kon_referenz_id', 'label' => 'Referenz ID'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListWeiBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'custom_wei_vorname_name_anrede', 'label' => 'Name', 'value' => $tableList['wei_anrede'] . ' ' . $tableList['wei_vorname'] . ' ' . $tableList['wei_name']];
    # $add[] = ['name' => 'wei_titel', 'label' => 'Titel'];
    # $add[] = ['name' => 'wei_anrede_brief', 'label' => 'Anrede Brief'];
    # $add[] = ['name' => 'wei_firma', 'label' => 'Firma'];
    # $add[] = ['name' => 'wei_zusatzfeld', 'label' => 'Zusatzfeld'];
    # $add[] = ['name' => 'wei_strasse', 'label' => 'Straße'];
    # $add[] = ['name' => 'wei_hausnummer', 'label' => 'Hausnummer'];
    # $add[] = ['name' => 'wei_plz', 'label' => 'PLZ'];
    # $add[] = ['name' => 'wei_ort', 'label' => 'Ort'];
    # $add[] = ['name' => 'wei_postfach', 'label' => 'Postfach'];
    # $add[] = ['name' => 'wei_postf_plz', 'label' => 'Postfach PLZ'];
    # $add[] = ['name' => 'wei_postf_ort', 'label' => 'Postfach Ort'];
    # $add[] = ['name' => 'wei_land_iso_land', 'label' => 'Land ISO Land'];
    # $add[] = ['name' => 'wei_email_zentrale', 'label' => 'E-Mail Zentrale'];
    $add[] = ['name' => 'custom_wei_email_direkt', 'label' => 'E-Mail-Adresse', 'value' => immoRenderLinkEmail($tableList['wei_email_direkt'], ['subject' => 'Betrifft Objekt ' . $tableList['vet_objektnr_extern']])];
    # $add[] = ['name' => 'wei_email_privat', 'label' => 'E-Mail privat'];
    # $add[] = ['name' => 'wei_email_sonstige', 'label' => 'E-Mail sonstige'];
    # $add[] = ['name' => 'wei_email_sonstige_emailart', 'label' => 'E-Mail sonstige Art'];
    # $add[] = ['name' => 'wei_email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung'];
    $add[] = ['name' => 'wei_tel_durchw', 'label' => 'Telefon (Durchwahl)'];
    # $add[] = ['name' => 'wei_tel_zentrale', 'label' => 'Telefon Zentrale'];
    $add[] = ['name' => 'wei_tel_handy', 'label' => 'Telefon (mobil)'];
    # $add[] = ['name' => 'wei_tel_fax', 'label' => 'Telefon Fax'];
    # $add[] = ['name' => 'wei_tel_privat', 'label' => 'Telefon privat'];
    # $add[] = ['name' => 'wei_tel_sonstige', 'label' => 'Telefon sonstige'];
    # $add[] = ['name' => 'wei_tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart'];
    # $add[] = ['name' => 'wei_tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung'];
    $add[] = ['name' => 'custom_wei_url', 'label' => 'Website', 'value' => immoRenderLinkWebsite($tableList['wei_url'], ['target' => '_blank', 'rel' => 'nofollow'])];
    # $add[] = ['name' => 'wei_adressfreigabe', 'label' => 'Adressfreigabe'];
    # $add[] = ['name' => 'wei_personennummer', 'label' => 'Personennummer'];
    # $add[] = ['name' => 'wei_freitextfeld', 'label' => 'Freitextfeld'];
    # $add[] = ['name' => 'wei_position', 'label' => 'Position'];
    # $add[] = ['name' => 'wei_email_feedback', 'label' => 'E-Mail Feedback'];
    # $add[] = ['name' => 'wei_immobilientreuhaenderid', 'label' => 'Immobilientreuhänder ID'];
    # $add[] = ['name' => 'wei_referenz_id', 'label' => 'Referenz ID'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListPreBasic(array $tableList) {

    $add = [];

    if (empty($tableList['pre_kaufpreis_auf_anfrage'])) {
        $add[] = ['name' => 'custom_pre_kaufpreis', 'label' => 'Kaufpreis', 'value' => $tableList['pre_kaufpreis'], 'format' => 'price'];
    } else {
        $add[] = ['name' => 'custom_pre_kaufpreis', 'label' => 'Kaufpreis', 'value' => 'auf Anfrage'];
    }

    # $add[] = ['name' => 'pre_kaufpreisnetto', 'label' => 'Kaufpreis netto'];
    # $add[] = ['name' => 'pre_kaufpreisnetto_kaufpreisust', 'label' => 'Kaufpreis netto USt'];
    # $add[] = ['name' => 'pre_kaufpreisbrutto', 'label' => 'Kaufpreis brutto'];
    # $add[] = ['name' => 'pre_nettokaltmiete', 'label' => 'Kaltmiete netto'];
    $add[] = ['name' => 'pre_kaltmiete', 'label' => 'Kaltmiete', 'format' => 'price'];
    $add[] = ['name' => 'pre_warmmiete', 'label' => 'Warmmiete', 'format' => 'price'];
    $add[] = ['name' => 'pre_nebenkosten', 'label' => 'Nebenkosten', 'format' => 'price'];
    # $add[] = ['name' => 'pre_heizkosten_enthalten', 'label' => 'Heizkosten enthalten'];
    # $add[] = ['name' => 'pre_heizkosten', 'label' => 'Heizkosten'];
    # $add[] = ['name' => 'pre_zzg_mehrwertsteuer', 'label' => 'Zuzüglich MwSt'];
    # $add[] = ['name' => 'pre_mietzuschlaege', 'label' => 'Mietzuschläge'];
    # $add[] = ['name' => 'pre_hauptmietzinsnetto', 'label' => 'Hauptmietzins netto'];
    # $add[] = ['name' => 'pre_hauptmietzinsnetto_hauptmietzinsust', 'label' => 'Hauptmietzins USt'];
    $add[] = ['name' => 'pre_pauschalmiete', 'label' => 'Pauschalmiete', 'format' => 'price'];
    # $add[] = ['name' => 'pre_betriebskostennetto', 'label' => 'Betriebskosten netto'];
    # $add[] = ['name' => 'pre_betriebskostennetto_betriebskostenust', 'label' => 'Betriebskosten USt'];
    # $add[] = ['name' => 'pre_evbnetto', 'label' => 'EVB netto', 'format' => 'price'];
    # $add[] = ['name' => 'pre_evbnetto_evbust', 'label' => 'EVB USt', 'format' => 'number', 'formatSymbol' => '%'];
    # $add[] = ['name' => 'pre_gesamtmietenetto', 'label' => 'Gesamtmiete netto'];
    # $add[] = ['name' => 'pre_gesamtmietenetto_gesamtmieteust', 'label' => 'Gesamtmiete USt'];
    # $add[] = ['name' => 'pre_gesamtbelastungbrutto', 'label' => 'Gesamtmiete brutto'];
    # $add[] = ['name' => 'pre_gesamtkostenprom2von', 'label' => 'Gesamtkosten pro m² von'];
    # $add[] = ['name' => 'pre_gesamtkostenprom2von_gesamtkostenprom2bis', 'label' => 'Gesamtkosten pro m² bis'];
    # $add[] = ['name' => 'pre_heizkostennetto', 'label' => 'Heizkosten netto'];
    # $add[] = ['name' => 'pre_heizkostennetto_heizkostenust', 'label' => 'Heizkosten USt'];
    # $add[] = ['name' => 'pre_monatlichekostennetto', 'label' => 'Monatliche Kosten netto'];
    # $add[] = ['name' => 'pre_monatlichekostennetto_monatlichekostenust', 'label' => 'Monatliche Kosten USt'];
    # $add[] = ['name' => 'pre_monatlichekostenbrutto', 'label' => 'Monatliche Kosten brutto'];
    # $add[] = ['name' => 'pre_nebenkostenprom2von', 'label' => 'Nebenkosten pro m² von'];
    # $add[] = ['name' => 'pre_nebenkostenprom2von_nebenkostenprom2bis', 'label' => 'Nebenkosten pro m² bis'];
    # $add[] = ['name' => 'pre_ruecklagenetto', 'label' => 'Rücklage netto'];
    # $add[] = ['name' => 'pre_ruecklagenetto_ruecklageust', 'label' => 'Rücklage USt'];
    # $add[] = ['name' => 'pre_sonstigekostennetto', 'label' => 'Sonstige Kosten netto'];
    # $add[] = ['name' => 'pre_ruecklagenetto_sonstigekostenust', 'label' => 'Sonstige Kosten USt'];
    # $add[] = ['name' => 'pre_sonstigemietenetto', 'label' => 'Sonstige Miete netto'];
    # $add[] = ['name' => 'pre_ruecklagenetto_sonstigemieteust', 'label' => 'Sonstige Miete USt'];
    # $add[] = ['name' => 'pre_summemietenetto', 'label' => 'Summe Miete netto'];
    # $add[] = ['name' => 'pre_summemietenetto_summemieteust', 'label' => 'Summe Miete USt'];
    # $add[] = ['name' => 'pre_nettomieteprom2von', 'label' => 'Nettomiete pro m² von'];
    # $add[] = ['name' => 'pre_summemietenetto_nettomieteprom2bis', 'label' => 'Nettomiete pro m² bis'];
    $add[] = ['name' => 'pre_pacht', 'label' => 'Pacht', 'format' => 'price'];
    $add[] = ['name' => 'pre_erbpacht', 'label' => 'Erbpacht', 'format' => 'price'];
    $add[] = ['name' => 'pre_hausgeld', 'label' => 'Hausgeld', 'format' => 'price'];
    $add[] = ['name' => 'pre_abstand', 'label' => 'Abstand', 'format' => 'price'];
    # $add[] = ['name' => 'pre_preis_zeitraum_von', 'label' => 'Preis Zeitraum von'];
    # $add[] = ['name' => 'pre_preis_zeitraum_bis', 'label' => 'Preis Zeitraum bis'];
    # $add[] = ['name' => 'pre_preis_zeiteinheit_zeiteinheit', 'label' => 'Preis Zeiteinheit'];
    $add[] = ['name' => 'pre_mietpreis_pro_qm', 'label' => 'Mietpreis pro m²', 'format' => 'price'];
    $add[] = ['name' => 'pre_kaufpreis_pro_qm', 'label' => 'Kaufpreis pro m²', 'format' => 'price'];
    # $add[] = ['name' => 'pre_provisionspflichtig', 'label' => 'Provisionspflichtig'];
    # $add[] = ['name' => 'pre_provision_teilen', 'label' => 'Provision teilen'];
    # $add[] = ['name' => 'pre_innen_courtage', 'label' => 'Innen Courtage'];
    # $add[] = ['name' => 'pre_innen_courtage_mit_mwst', 'label' => 'Innen Courtage mit MwSt'];
    $add[] = ['name' => 'pre_aussen_courtage', 'label' => 'Courtage', 'labelAppend' => '%'];
    # $add[] = ['name' => 'pre_aussen_courtage_mit_mwst', 'label' => 'Außen Courtage mit MwSt'];
    $add[] = ['name' => 'pre_courtage_hinweis', 'label' => 'Courtage Hinweis'];
    # $add[] = ['name' => 'pre_provisionnetto', 'label' => 'Provision netto'];
    # $add[] = ['name' => 'pre_provisionbrutto', 'label' => 'Provision brutto'];
    # $add[] = ['name' => 'pre_waehrung_iso_waehrung', 'label' => 'Währung ISO Währung'];
    # $add[] = ['name' => 'pre_mwst_satz', 'label' => 'MwSt Satz'];
    # $add[] = ['name' => 'pre_mwst_gesamt', 'label' => 'MwSt Gesamt'];
    $add[] = ['name' => 'pre_freitext_preis', 'label' => 'Hinweise zum Preis'];
    # $add[] = ['name' => 'pre_x_fache', 'label' => 'X-fache'];
    # $add[] = ['name' => 'pre_nettorendite', 'label' => 'Nettorendite'];
    # $add[] = ['name' => 'pre_nettorendite_soll', 'label' => 'Nettorendite Soll'];
    # $add[] = ['name' => 'pre_nettorendite_ist', 'label' => 'Nettorendite Ist'];
    # $add[] = ['name' => 'pre_mieteinnahmen_ist', 'label' => 'Mieteinnahmen Ist'];
    # $add[] = ['name' => 'pre_mieteinnahmen_ist_periode', 'label' => 'Mieteinnahmen Ist Periode'];
    # $add[] = ['name' => 'pre_mieteinnahmen_soll', 'label' => 'Mieteinnahmen Soll'];
    # $add[] = ['name' => 'pre_mieteinnahmen_soll_periode', 'label' => 'Mieteinnahmen Soll Periode'];
    $add[] = ['name' => 'pre_erschliessungskosten', 'label' => 'Erschließungskosten', 'format' => 'price'];
    $add[] = ['name' => 'pre_kaution', 'label' => 'Kaution', 'format' => 'price'];
    $add[] = ['name' => 'pre_kaution_text', 'label' => 'Hinweise zur Kaution'];
    # $add[] = ['name' => 'pre_geschaeftsguthaben', 'label' => 'Geschäftsguthaben'];
    $add[] = ['name' => 'pre_stp_carport_stellplatzmiete', 'label' => 'Stellplatz Carport Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_carport_stellplatzkaufpreis', 'label' => 'Stellplatz Carport Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_carport_anzahl', 'label' => 'Stellplatz Carport Anzahl', 'format' => 'number'];
    $add[] = ['name' => 'pre_stp_duplex_stellplatzmiete', 'label' => 'Stellplatz Duplex Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_duplex_stellplatzkaufpreis', 'label' => 'Stellplatz Duplex Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_duplex_anzahl', 'label' => 'Stellplatz Duplex Anzahl', 'format' => 'number'];
    $add[] = ['name' => 'pre_stp_freiplatz_stellplatzmiete', 'label' => 'Stellplatz Freiplatz Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_freiplatz_stellplatzkaufpreis', 'label' => 'Stellplatz Freiplatz Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_freiplatz_anzahl', 'label' => 'Stellplatz Freiplatz Anzahl', 'format' => 'number'];
    $add[] = ['name' => 'pre_stp_garage_stellplatzmiete', 'label' => 'Stellplatz Garage Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_garage_stellplatzkaufpreis', 'label' => 'Stellplatz Garage Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_garage_anzahl', 'label' => 'Stellplatz Garage Anzahl', 'format' => 'number'];
    $add[] = ['name' => 'pre_stp_parkhaus_stellplatzmiete', 'label' => 'Stellplatz Parkhaus Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_parkhaus_stellplatzkaufpreis', 'label' => 'Stellplatz Parkhaus Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_parkhaus_anzahl', 'label' => 'Stellplatz Parkhaus Anzahl', 'format' => 'number'];
    $add[] = ['name' => 'pre_stp_tiefgarage_stellplatzmiete', 'label' => 'Stellplatz Tiefgarage Miete', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_tiefgarage_stellplatzkaufpreis', 'label' => 'Stellplatz Tiefgarage Kaufpreis', 'format' => 'price'];
    $add[] = ['name' => 'pre_stp_tiefgarage_anzahl', 'label' => 'Stellplatz Tiefgarage Anzahl', 'format' => 'number'];
    # $add[] = ['name' => 'pre_stp_sonstige_stellplatzmiete', 'label' => 'Stellplatz Sonstige Miete'];
    # $add[] = ['name' => 'pre_stp_sonstige_stellplatzkaufpreis', 'label' => 'Stellplatz Sonstige Kaufpreis'];
    # $add[] = ['name' => 'pre_stp_sonstige_anzahl', 'label' => 'Stellplatz Sonstige Anzahl'];
    # $add[] = ['name' => 'pre_stp_sonstige_platzart', 'label' => 'Stellplatz Sonstige Platzart'];
    # $add[] = ['name' => 'pre_stp_sonstige_bemerkung', 'label' => 'Stellplatz Sonstige Bemerkung'];
    # $add[] = ['name' => 'pre_richtpreis', 'label' => 'Richtpreis'];
    # $add[] = ['name' => 'pre_richtpreisprom2', 'label' => 'Richtpreis pro m²'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListBieBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'bie_beginn_angebotsphase', 'label' => 'Beginn Angebotsphase'];
    # $add[] = ['name' => 'bie_besichtigungstermin', 'label' => 'Besichtigungstermin'];
    # $add[] = ['name' => 'bie_besichtigungstermin_2', 'label' => 'Besichtigungstermin 2'];
    # $add[] = ['name' => 'bie_beginn_bietzeit', 'label' => 'Beginn Bietzeit'];
    # $add[] = ['name' => 'bie_ende_bietzeit', 'label' => 'Ende Bietzeit'];
    # $add[] = ['name' => 'bie_hoechstgebot_zeigen', 'label' => 'Höchstgebot zeigen'];
    # $add[] = ['name' => 'bie_mindestpreis', 'label' => 'Mindestpreis'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListVerBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'ver_zwangsversteigerung', 'label' => 'Zwangsversteigerung'];
    # $add[] = ['name' => 'ver_aktenzeichen', 'label' => 'Aktenzeichen'];
    # $add[] = ['name' => 'ver_zvtermin', 'label' => 'ZV Termin'];
    # $add[] = ['name' => 'ver_zusatztermin', 'label' => 'Zusatztermin'];
    # $add[] = ['name' => 'ver_amtsgericht', 'label' => 'Amtsgericht'];
    # $add[] = ['name' => 'ver_verkehrswert', 'label' => 'Verkehrswert'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListFlaBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'fla_wohnflaeche', 'label' => 'Wohnfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_nutzflaeche', 'label' => 'Nutzfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_gesamtflaeche', 'label' => 'Gesamtfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_ladenflaeche', 'label' => 'Ladenfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_lagerflaeche', 'label' => 'Lagerfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_verkaufsflaeche', 'label' => 'Verkaufsfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_freiflaeche', 'label' => 'Freifläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_bueroflaeche', 'label' => 'Bürofläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_bueroteilflaeche', 'label' => 'Büroteilfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_fensterfront', 'label' => 'Fensterfront Länge', 'format' => 'space', 'formatSymbol' => 'm'];
    $add[] = ['name' => 'fla_verwaltungsflaeche', 'label' => 'Verwaltungsfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_gastroflaeche', 'label' => 'Gastrofläche', 'format' => 'space'];
    # $add[] = ['name' => 'fla_grz', 'label' => 'GRZ'];
    # $add[] = ['name' => 'fla_gfz', 'label' => 'GFZ'];
    # $add[] = ['name' => 'fla_bmz', 'label' => 'BMZ'];
    # $add[] = ['name' => 'fla_bgf', 'label' => 'BGF'];
    $add[] = ['name' => 'fla_grundstuecksflaeche', 'label' => 'Grundstücksfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_sonstflaeche', 'label' => 'Sonstige Fläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_anzahl_zimmer', 'label' => 'Anzahl Zimmer', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_schlafzimmer', 'label' => 'Anzahl Schlafzimmer', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_badezimmer', 'label' => 'Anzahl Badezimmer', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_sep_wc', 'label' => 'Anzahl separate WCs', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_balkone', 'label' => 'Anzahl Balkone', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_terrassen', 'label' => 'Anzahl Terrassen', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_logia', 'label' => 'Anzahl Loggien', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_balkon_terrasse_flaeche', 'label' => 'Fläche aller Balkone/Terrassen', 'format' => 'space'];
    $add[] = ['name' => 'fla_anzahl_wohn_schlafzimmer', 'label' => 'Anzahl Wohn-Schlafzimmer', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_gartenflaeche', 'label' => 'Gartenfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_kellerflaeche', 'label' => 'Kellerfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_fensterfront_qm', 'label' => 'Fensterfrontfläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_grundstuecksfront', 'label' => 'Grundstücksfront', 'format' => 'space', 'formatSymbol' => 'm'];
    $add[] = ['name' => 'fla_dachbodenflaeche', 'label' => 'Dachbodenfläche', 'format' => 'space'];
    # $add[] = ['name' => 'fla_teilbar_ab', 'label' => 'Teilbar ab'];
    # $add[] = ['name' => 'fla_beheizbare_flaeche', 'label' => 'Beheizbare Fläche'];
    $add[] = ['name' => 'fla_anzahl_stellplaetze', 'label' => 'Anzahl Stellplätze', 'format' => 'number'];
    # $add[] = ['name' => 'fla_plaetze_gastraum', 'label' => 'Plätze Gastraum'];
    # $add[] = ['name' => 'fla_anzahl_betten', 'label' => 'Anzahl Betten'];
    # $add[] = ['name' => 'fla_anzahl_tagungsraeume', 'label' => 'Anzahl Tagungsräume'];
    $add[] = ['name' => 'fla_vermietbare_flaeche', 'label' => 'Vermietbare Fläche', 'format' => 'space'];
    $add[] = ['name' => 'fla_anzahl_wohneinheiten', 'label' => 'Anzahl Wohneinheiten', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_anzahl_gewerbeeinheiten', 'label' => 'Anzahl Gewerbeeinheiten', 'format' => 'number', 'formatDigits' => 1];
    $add[] = ['name' => 'fla_einliegerwohnung', 'label' => 'Einliegerwohnung'];
    # $add[] = ['name' => 'fla_kubatur', 'label' => 'Kubatur'];
    # $add[] = ['name' => 'fla_ausnuetzungsziffer', 'label' => 'Ausnützungsziffer'];
    $add[] = ['name' => 'fla_flaechevon', 'label' => 'Fläche von', 'format' => 'space'];
    $add[] = ['name' => 'fla_flaechebis', 'label' => 'Fläche bis', 'format' => 'space'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListAusBasic(array $tableList) {

    $add = [];

    $custom_aus_ausstatt_kategorie = immoGetOptionsString(strtoupper($tableList['aus_ausstatt_kategorie']), [
        'STANDARD' => 'Standard',
        'GEHOBEN' => 'Gehoben',
        'LUXUS' => 'Luxus'
    ]);

    $add[] = ['name' => 'custom_aus_ausstatt_kategorie', 'label' => 'Kategorie der Ausstattung', 'value' => $custom_aus_ausstatt_kategorie];
    $add[] = ['name' => 'aus_wg_geeignet', 'label' => 'Für WG geeignet'];
    $add[] = ['name' => 'aus_raeume_veraenderbar', 'label' => 'Räume veränderbar'];
    $add[] = ['name' => 'aus_bad_dusche', 'label' => 'Dusche'];
    $add[] = ['name' => 'aus_bad_wanne', 'label' => 'Wanne'];
    $add[] = ['name' => 'aus_bad_fenster', 'label' => 'Bad mit Fenster'];
    $add[] = ['name' => 'aus_bad_bidet', 'label' => ' Bidet'];
    $add[] = ['name' => 'aus_bad_pissoir', 'label' => 'Pissoir'];
    $add[] = ['name' => 'aus_kueche_ebk', 'label' => 'Einbauküche'];
    $add[] = ['name' => 'aus_kueche_offen', 'label' => 'Offene Küche'];
    $add[] = ['name' => 'aus_kueche_pantry', 'label' => 'Pantryküche'];
    $add[] = ['name' => 'aus_boden_fliesen', 'label' => 'Boden Fliesen'];
    $add[] = ['name' => 'aus_boden_stein', 'label' => 'Boden Stein'];
    $add[] = ['name' => 'aus_boden_teppich', 'label' => 'Boden Teppich'];
    $add[] = ['name' => 'aus_boden_parkett', 'label' => 'Boden Parkett'];
    $add[] = ['name' => 'aus_boden_fertigparkett', 'label' => 'Boden Fertigparkett'];
    $add[] = ['name' => 'aus_boden_laminat', 'label' => 'Boden Laminat'];
    $add[] = ['name' => 'aus_boden_dielen', 'label' => 'Boden Dielen'];
    $add[] = ['name' => 'aus_boden_kunststoff', 'label' => 'Boden Kunststoff'];
    $add[] = ['name' => 'aus_boden_estrich', 'label' => 'Boden Estrich'];
    $add[] = ['name' => 'aus_boden_doppelboden', 'label' => 'Boden Doppelboden'];
    $add[] = ['name' => 'aus_boden_linoleum', 'label' => 'Boden Linoleum'];
    $add[] = ['name' => 'aus_boden_marmor', 'label' => 'Boden Marmor'];
    $add[] = ['name' => 'aus_boden_terrakotta', 'label' => 'Boden Terrakotta'];
    $add[] = ['name' => 'aus_boden_granit', 'label' => 'Boden Granit'];
    $add[] = ['name' => 'aus_kamin', 'label' => 'Offener Kamin vorhanden'];
    $add[] = ['name' => 'aus_heizungsart_ofen', 'label' => 'Ofenheizung'];
    $add[] = ['name' => 'aus_heizungsart_etage', 'label' => 'Etagenheizung'];
    $add[] = ['name' => 'aus_heizungsart_zentral', 'label' => 'Zentralheizung'];
    $add[] = ['name' => 'aus_heizungsart_fern', 'label' => 'Fernheizung'];
    $add[] = ['name' => 'aus_heizungsart_fussboden', 'label' => 'Fußbodenheizung'];
    $add[] = ['name' => 'aus_befeuerung_oel', 'label' => 'Befeuerung mit Öl'];
    $add[] = ['name' => 'aus_befeuerung_gas', 'label' => 'Befeuerung mit Gas'];
    $add[] = ['name' => 'aus_befeuerung_elektro', 'label' => 'Befeuerung mit Strom'];
    $add[] = ['name' => 'aus_befeuerung_alternativ', 'label' => 'Befeuerung alternativ'];
    $add[] = ['name' => 'aus_befeuerung_solar', 'label' => 'Befeuerung mit Solarenergie'];
    $add[] = ['name' => 'aus_befeuerung_erdwaerme', 'label' => 'Befeuerung mit Erdwärme'];
    $add[] = ['name' => 'aus_befeuerung_luftwp', 'label' => 'Befeuerung mit Luftwärmepumpe'];
    $add[] = ['name' => 'aus_befeuerung_fern', 'label' => 'Befeuerung mit Fernwärme'];
    $add[] = ['name' => 'aus_befeuerung_block', 'label' => 'Befeuerung mit Blockheizkraftwerk'];
    $add[] = ['name' => 'aus_befeuerung_wasser_elektro', 'label' => 'Ergänzendes dezentrales Warmwasser'];
    $add[] = ['name' => 'aus_befeuerung_pellet', 'label' => 'Befeuerung mit Pellets'];
    $add[] = ['name' => 'aus_befeuerung_kohle', 'label' => 'Befeuerung mit Kohle'];
    $add[] = ['name' => 'aus_befeuerung_holz', 'label' => 'Befeuerung mit Holz'];
    $add[] = ['name' => 'aus_befeuerung_fluessiggas', 'label' => 'Befeuerung mit Flüssiggas'];
    $add[] = ['name' => 'aus_klimatisiert', 'label' => 'Klimatisiert'];
    $add[] = ['name' => 'aus_fahrstuhl_personen', 'label' => 'Personenfahrstuhl'];
    $add[] = ['name' => 'aus_fahrstuhl_lasten', 'label' => 'Lastenaufzug'];
    $add[] = ['name' => 'aus_stellplatzart_garage', 'label' => 'Stellplatz in Garage'];
    $add[] = ['name' => 'aus_stellplatzart_tiefgarage', 'label' => 'Stellplatz in Tiefgarage'];
    $add[] = ['name' => 'aus_stellplatzart_carport', 'label' => 'Stellplatz im Carport'];
    $add[] = ['name' => 'aus_stellplatzart_freiplatz', 'label' => 'Stellplatz im Freien'];
    $add[] = ['name' => 'aus_stellplatzart_parkhaus', 'label' => 'Stellplatz im Parkhaus'];
    $add[] = ['name' => 'aus_stellplatzart_duplex', 'label' => 'Stellplatz Duplex'];
    $add[] = ['name' => 'aus_gartennutzung', 'label' => 'Gartennutzung'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_nord', 'label' => 'Ausrichtung Balkon Terrasse Nord'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_ost', 'label' => 'Ausrichtung Balkon Terrasse Ost'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_sued', 'label' => 'Ausrichtung Balkon Terrasse Süd'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_west', 'label' => 'Ausrichtung Balkon Terrasse West'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_nordost', 'label' => 'Ausrichtung Balkon Terrasse Nordost'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_nordwest', 'label' => 'Ausrichtung Balkon Terrasse Nordest'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_suedost', 'label' => 'Ausrichtung Balkon Terrasse Südost'];
    # $add[] = ['name' => 'aus_ausricht_balkon_terrasse_suedwest', 'label' => 'Ausrichtung Balkon Terrasse Südwest'];

    $custom_aus_moebliert_moeb = immoGetOptionsString(strtoupper($tableList['aus_moebliert_moeb']), [
        'VOLL' => 'Vollmöbliert',
        'TEIL' => 'Teilmöbliert'
    ]);

    $add[] = ['name' => 'custom_aus_moebliert_moeb', 'label' => 'Möblierung', 'value' => $custom_aus_moebliert_moeb];
    $add[] = ['name' => 'aus_rollstuhlgerecht', 'label' => 'Rollstuhlgerecht'];
    $add[] = ['name' => 'aus_kabel_sat_tv', 'label' => 'Kabel oder Sat-TV'];
    $add[] = ['name' => 'aus_dvbt', 'label' => 'DVBT-Empfang'];
    $add[] = ['name' => 'aus_barrierefrei', 'label' => 'Barrierefrei'];
    $add[] = ['name' => 'aus_sauna', 'label' => 'Sauna'];
    $add[] = ['name' => 'aus_swimmingpool', 'label' => 'Swimmingpool'];
    $add[] = ['name' => 'aus_wasch_trockenraum', 'label' => 'Wasch- und Trockenraum'];
    $add[] = ['name' => 'aus_wintergarten', 'label' => 'Wintergarten'];
    $add[] = ['name' => 'aus_dv_verkabelung', 'label' => 'DV-Verkabelung'];
    $add[] = ['name' => 'aus_rampe', 'label' => 'Rampe'];
    $add[] = ['name' => 'aus_hebebuehne', 'label' => 'Hebebühne'];
    $add[] = ['name' => 'aus_kran', 'label' => 'Kran'];
    $add[] = ['name' => 'aus_gastterrasse', 'label' => 'Gastterrasse'];
    # $add[] = ['name' => 'aus_stromanschlusswert', 'label' => 'Stromanschlusswert'];
    # $add[] = ['name' => 'aus_kantine_cafeteria', 'label' => 'Kantine Cafeteria'];
    # $add[] = ['name' => 'aus_teekueche', 'label' => 'Teeküche'];
    $add[] = ['name' => 'aus_hallenhoehe', 'label' => 'Hallenhöhe', 'format' => 'number', 'formatSymbol' => 'm'];
    # $add[] = ['name' => 'aus_angeschl_gastronomie_hotelrestaurant', 'label' => 'Angeschlossene Gastronomie Hotelrestaurant'];
    # $add[] = ['name' => 'aus_angeschl_gastronomie_bar', 'label' => 'Angeschlossene Gastronomie Bar'];
    # $add[] = ['name' => 'aus_brauereibindung', 'label' => 'Brauereibindung'];
    # $add[] = ['name' => 'aus_sporteinrichtungen', 'label' => 'Sporteinrichtungen'];
    # $add[] = ['name' => 'aus_wellnessbereich', 'label' => 'Wellnessbereich'];
    # $add[] = ['name' => 'aus_serviceleistungen_betreutes_wohnen', 'label' => 'Serviceleistungen Betreutes Wohnen'];
    # $add[] = ['name' => 'aus_serviceleistungen_catering', 'label' => 'Serviceleistungen Catering'];
    # $add[] = ['name' => 'aus_serviceleistungen_reinigung', 'label' => 'Serviceleistungen Reinigung'];
    # $add[] = ['name' => 'aus_serviceleistungen_einkauf', 'label' => 'Serviceleistungen Einkauf'];
    # $add[] = ['name' => 'aus_serviceleistungen_wachdienst', 'label' => 'Serviceleistungen Wachdienst'];
    # $add[] = ['name' => 'aus_telefon_ferienimmobilie', 'label' => 'Telefon Ferienimmobilie'];
    $add[] = ['name' => 'aus_breitband_zugang_art', 'label' => 'Breitband Art'];
    $add[] = ['name' => 'aus_breitband_zugang_speed', 'label' => 'Breitband Geschwindigkeit', 'format' => 'number', 'formatSymbol' => 'Mbit'];
    $add[] = ['name' => 'aus_umts_empfang', 'label' => 'UMTS-Empfang'];
    $add[] = ['name' => 'aus_sicherheitstechnik_alarmanlage', 'label' => 'Alarmanlage'];
    $add[] = ['name' => 'aus_sicherheitstechnik_kamera', 'label' => 'Kamera'];
    $add[] = ['name' => 'aus_sicherheitstechnik_polizeiruf', 'label' => 'Polizeiruf'];

    $custom_aus_unterkellert_keller = immoGetOptionsString(strtoupper($tableList['aus_unterkellert_keller']), [
        'JA' => 'ja',
        'NEIN' => 'nein',
        'TEIL' => 'teilweise'
    ]);

    $add[] = ['name' => 'custom_aus_unterkellert_keller', 'label' => 'Unterkellerung', 'value' => $custom_aus_unterkellert_keller];
    $add[] = ['name' => 'aus_abstellraum', 'label' => 'Abstellraum'];
    $add[] = ['name' => 'aus_fahrradraum', 'label' => 'Fahrradraum'];
    $add[] = ['name' => 'aus_rolladen', 'label' => 'Rolladen'];
    $add[] = ['name' => 'aus_dachform_krueppelwalmdach', 'label' => 'Krüppelwalmdach'];
    $add[] = ['name' => 'aus_dachform_mansarddach', 'label' => 'Mansarddach'];
    $add[] = ['name' => 'aus_dachform_pultdach', 'label' => 'Pultdach'];
    $add[] = ['name' => 'aus_dachform_satteldach', 'label' => 'Satteldach'];
    $add[] = ['name' => 'aus_dachform_walmdach', 'label' => 'Walmdach'];
    $add[] = ['name' => 'aus_dachform_flachdach', 'label' => 'Flachdach'];
    $add[] = ['name' => 'aus_dachform_pyramidendach', 'label' => 'Pyramidendach'];
    $add[] = ['name' => 'aus_bauweise_massiv', 'label' => 'Massivbauweise'];
    $add[] = ['name' => 'aus_bauweise_fertigteile', 'label' => 'Fertigteilbauweise'];
    $add[] = ['name' => 'aus_bauweise_holz', 'label' => 'Holzbauweise'];
    $add[] = ['name' => 'aus_ausbaustufe_bausatzhaus', 'label' => 'Bausatzhaus'];
    $add[] = ['name' => 'aus_ausbaustufe_ausbauhaus', 'label' => 'Ausbauhaus'];
    $add[] = ['name' => 'aus_ausbaustufe_schluesselfertigmitkeller', 'label' => 'Schlüsselfertig mit Keller'];
    $add[] = ['name' => 'aus_ausbaustufe_schluesselfertigohnebodenplatte', 'label' => 'Schlüsselfertig ohne Bodenplatte'];
    $add[] = ['name' => 'aus_ausbaustufe_schluesselfertigmitbodenplatte', 'label' => 'Schlüsselfertig mit Bodenplatte'];
    $add[] = ['name' => 'aus_energietyp_passivhaus', 'label' => 'Passivhaus'];
    $add[] = ['name' => 'aus_energietyp_niedrigenergie', 'label' => 'Niedrigenergiehaus'];
    $add[] = ['name' => 'aus_energietyp_neubaustandard', 'label' => 'Neubaustandard'];
    $add[] = ['name' => 'aus_energietyp_kfw40', 'label' => 'Energietyp Kfw 40'];
    $add[] = ['name' => 'aus_energietyp_kfw60', 'label' => 'Energietyp Kfw 60'];
    $add[] = ['name' => 'aus_energietyp_kfw55', 'label' => 'Energietyp Kfw 55'];
    $add[] = ['name' => 'aus_energietyp_kfw70', 'label' => 'Energietyp Kfw 70'];
    $add[] = ['name' => 'aus_energietyp_minergiebauweise', 'label' => 'Bauweise Minergie'];
    $add[] = ['name' => 'aus_energietyp_minergie_zertifiziert', 'label' => 'Minergie zertifiziert'];
    $add[] = ['name' => 'aus_bibliothek', 'label' => 'Bibliothek'];
    $add[] = ['name' => 'aus_dachboden', 'label' => 'Dachboden'];
    $add[] = ['name' => 'aus_gaestewc', 'label' => 'Gäste WC'];
    $add[] = ['name' => 'aus_kabelkanaele', 'label' => 'Kabelkanäle'];
    $add[] = ['name' => 'aus_seniorengerecht', 'label' => 'Seniorengerecht'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListZusBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'zus_baujahr', 'label' => 'Baujahr'];
    $add[] = ['name' => 'zus_letztemodernisierung', 'label' => 'Letzte Modernisierung'];

    $custom_zus_zustand_zustand_art = immoGetOptionsString(strtoupper($tableList['zus_zustand_zustand_art']), [
        'ERSTBEZUG' => 'Erstbezug',
        'TEIL_VOLLRENOVIERUNGSBED' => 'teil- oder vollrenovierungsbedürftig',
        'NEUWERTIG' => 'neuwertig',
        'TEIL_VOLLSANIERT' => 'teil- oder vollsaniert',
        'TEIL_VOLLRENOVIERT' => 'teil- oder vollrenoviert',
        'TEIL_SANIERT' => 'teilsaniert',
        'VOLL_SANIERT' => 'vollsaniert',
        'SANIERUNGSBEDUERFTIG' => 'sanierungsbedürftig',
        'BAUFAELLIG' => 'baufällig',
        'NACH_VEREINBARUNG' => 'nach Vereinbarung',
        'MODERNISIERT' => 'modernisiert',
        'GEPFLEGT' => 'gepflegt',
        'ROHBAU' => 'Rohbau',
        'ENTKERNT' => 'entkernt',
        'ABRISSOBJEKT' => 'Abrissobjekt',
        'PROJEKTIERT' => 'projektiert'
    ]);

    $add[] = ['name' => 'custom_zus_zustand_zustand_art', 'label' => 'Zustand', 'value' => $custom_zus_zustand_zustand_art];

    $custom_zus_alter_alter_attr = immoGetOptionsString(strtoupper($tableList['zus_alter_alter_attr']), [
        'ALTBAU' => 'Altbau',
        'NEUBAU' => 'Neubau'
    ]);

    $add[] = ['name' => 'custom_zus_alter_alter_attr', 'label' => 'Neu- oder Altbau', 'value' => $custom_zus_alter_alter_attr];

    $custom_zus_bebaubar_nach_bebaubar_attr = immoGetOptionsString(strtoupper($tableList['zus_bebaubar_nach_bebaubar_attr']), [
        '34_NACHBARSCHAFT' => 'Nachbarbebauung §34',
        '35_AUSSENGEBIET' => 'Außengebiet §35',
        'B_PLAN' => 'Bebauungsplan',
        'KEIN BAULAND' => 'kein Bauland',
        'BAUERWARTUNGSLAND' => 'Bauerwartungsland',
        'LAENDERSPEZIFISCH' => 'länderspezifisch',
        'BAULAND_OHNE_B_PLAN' => 'Bauland ohne Bebauungsplan'
    ]);

    $add[] = ['name' => 'custom_zus_bebaubar_nach_bebaubar_attr', 'label' => 'Bebaubarkeit', 'value' => $custom_zus_bebaubar_nach_bebaubar_attr];

    $custom_zus_erschliessung_erschl_attr = immoGetOptionsString(strtoupper($tableList['zus_erschliessung_erschl_attr']), [
        'UNERSCHLOSSEN' => 'unerschlossen',
        'TEILERSCHLOSSEN' => 'teilerschlossen',
        'VOLLERSCHLOSSEN' => 'vollerschlossen',
        'ORTSUEBLICHERSCHLOSSEN' => 'ortsüblich erschlossen'
    ]);

    $add[] = ['name' => 'custom_zus_erschliessung_erschl_attr', 'label' => 'Erschließung', 'value' => $custom_zus_erschliessung_erschl_attr];

    $custom_zus_erschliessung_umfang_erschl_attr = immoGetOptionsString(strtoupper($tableList['zus_erschliessung_umfang_erschl_attr']), [
        'GAS' => 'Gas',
        'WASSER' => 'Wasser',
        'STROM' => 'Strom',
        'TK' => 'Telekommunikation'
    ]);

    $add[] = ['name' => 'custom_zus_erschliessung_umfang_erschl_attr', 'label' => 'Erschließung Umfang', 'value' => $custom_zus_erschliessung_umfang_erschl_attr];

    $add[] = ['name' => 'zus_bauzone', 'label' => 'Bauzone'];
    $add[] = ['name' => 'zus_altlasten', 'label' => 'Altlasten'];
    # $add[] = ['name' => 'zus_energiepass_epart', 'label' => 'Energiepass epart'];
    # $add[] = ['name' => 'zus_energiepass_gueltig_bis', 'label' => 'Energiepass gültig bis'];
    # $add[] = ['name' => 'zus_energiepass_energieverbrauchkennwert', 'label' => 'Energiepass Energieverbrauchkennwert'];
    # $add[] = ['name' => 'zus_energiepass_mitwarmwasser', 'label' => 'Energiepass mit Warmwasser'];
    # $add[] = ['name' => 'zus_energiepass_endenergiebedarf', 'label' => 'Energiepass Endenergiebedarf'];
    # $add[] = ['name' => 'zus_energiepass_primaerenergietraeger', 'label' => 'Energiepass Primärenergieträger'];
    # $add[] = ['name' => 'zus_energiepass_stromwert', 'label' => 'Energiepass Stromwert'];
    # $add[] = ['name' => 'zus_energiepass_waermewert', 'label' => 'Energiepass Wärmewert'];
    # $add[] = ['name' => 'zus_energiepass_wertklasse', 'label' => 'Energiepass Wertklasse'];
    # $add[] = ['name' => 'zus_energiepass_baujahr', 'label' => 'Energiepass Baujahr'];
    # $add[] = ['name' => 'zus_energiepass_ausstelldatum', 'label' => 'Energiepass Ausstelldatum'];
    # $add[] = ['name' => 'zus_energiepass_jahrgang', 'label' => 'Energiepass Jahrgang'];
    # $add[] = ['name' => 'zus_energiepass_gebaeudeart', 'label' => 'Energiepass Gebäudeart'];
    # $add[] = ['name' => 'zus_energiepass_epasstext', 'label' => 'Energiepass Epass Text'];
    # $add[] = ['name' => 'zus_energiepass_geg2018', 'label' => 'Energiepass GEG 2018'];
    # $add[] = ['name' => 'zus_hwbwert', 'label' => 'Heizwärmebedarf Wert'];
    # $add[] = ['name' => 'zus_hwbklasse', 'label' => 'Heizwärmebedarf Klasse'];
    # $add[] = ['name' => 'zus_fgeewert', 'label' => 'Gesamtenergieeffizienz-Faktor Wert'];
    # $add[] = ['name' => 'zus_fgeeklasse', 'label' => 'Gesamtenergieeffizienz-Faktor Klasse'];
    # $add[] = ['name' => 'zus_verkaufstatus_stand', 'label' => 'Verkaufstatus Stand'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListBewBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'bew_feld_name', 'label' => 'Name'];
    # $add[] = ['name' => 'bew_feld_wert', 'label' => 'Wert'];
    # $add[] = ['name' => 'bew_feld_typ', 'label' => 'Typ'];
    # $add[] = ['name' => 'bew_feld_modus', 'label' => 'Modus'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListInfBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'inf_zulieferung', 'label' => 'Anlieferungen per LKW möglich'];

    $custom_inf_ausblick_blick = immoGetOptionsString(strtoupper($tableList['inf_ausblick_blick']), [
        'FERNE' => 'Fernblick',
        'SEE' => 'Seeblick',
        'BERGE' => 'Bergblick',
        'MEER' => 'Meerblick'
    ]);

    $add[] = ['name' => 'custom_inf_ausblick_blick', 'label' => 'Aussicht', 'value' => $custom_inf_ausblick_blick];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_autobahn', 'label' => 'Distanz zur Autobahn', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_bus', 'label' => 'Distanz zum Bus', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_fernbahnhof', 'label' => 'Distanz zum Fernbahnhof', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_flughafen', 'label' => 'Distanz zum Flughafen', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_gesamtschule', 'label' => 'Distanz zur Gesamtschule', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_grundschule', 'label' => 'Distanz zur Grundschule', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_gymnasium', 'label' => 'Distanz zum Gymnasium', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_hauptschule', 'label' => 'Distanz zur Hauptschule', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_kindergaerten', 'label' => 'Distanz zur Kindertagesstätte', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_realschule', 'label' => 'Distanz zur Realschule', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_us_bahn', 'label' => 'Distanz zum Nahbahnhof (ÖPNV)', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_zentrum', 'label' => 'Distanz zum Zentrum', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_einkaufsmoeglichkeiten', 'label' => 'Distanz zu Einkaufsmöglichkeiten', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_distanz_zu_gaststaetten', 'label' => 'Distanz zu Gaststätten', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_meer', 'label' => 'Distanz zum Meer', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_naherholung', 'label' => 'Distanz zur Naherholung', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_see', 'label' => 'Distanz zum See', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_skigebiet', 'label' => 'Distanz zum Skigebiet', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_sportanlagen', 'label' => 'Distanz zu Sportanlagen', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_strand', 'label' => 'Distanz zum Strand', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];
    $add[] = ['name' => 'inf_distanzen_sport_distanz_zu_sport_wandergebiete', 'label' => 'Distanz zum Wandergebiet', 'format' => 'space', 'formatSymbol' => 'km', 'labelPrepend' => 'ca.'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListFreBasic(array $tableList) {

    $add = [];

    $add[] = ['name' => 'fre_objekttitel', 'label' => 'Objekttitel'];
    $add[] = ['name' => 'fre_dreizeiler', 'label' => 'Dreizeiler'];
    $add[] = ['name' => 'fre_lage', 'label' => 'Lage'];
    $add[] = ['name' => 'fre_ausstatt_beschr', 'label' => 'Beschreibung der Ausstattung'];
    $add[] = ['name' => 'fre_objektbeschreibung', 'label' => 'Objektbeschreibung'];
    $add[] = ['name' => 'fre_sonstige_angaben', 'label' => 'Sonstige Angaben'];
    $add[] = ['name' => 'fre_objekt_text', 'label' => 'Objekt Text'];
    $add[] = ['name' => 'fre_objekt_text_lang', 'label' => 'Objekt Langtext'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListVeoBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'veo_objektadresse_freigeben', 'label' => 'Objektadresse freigeben'];
    $add[] = ['name' => 'veo_verfuegbar_ab', 'label' => 'Objekt verfügbar ab'];
    # $add[] = ['name' => 'veo_abdatum', 'label' => 'Ab Datum'];
    # $add[] = ['name' => 'veo_bisdatum', 'label' => 'Bis Datum'];
    $add[] = ['name' => 'veo_min_mietdauer_min_dauer', 'label' => 'Mietdauer mindestens'];
    $add[] = ['name' => 'veo_max_mietdauer_max_dauer', 'label' => 'Mietdauer maximal'];
    # $add[] = ['name' => 'veo_versteigerungstermin', 'label' => 'Versteigerungstermin'];
    $add[] = ['name' => 'veo_wbs_sozialwohnung', 'label' => 'Sozialwohnung (WBS)'];
    $add[] = ['name' => 'veo_vermietet', 'label' => 'Aktuell vermietet'];
    # $add[] = ['name' => 'veo_gruppennummer', 'label' => 'Gruppennummer'];
    $add[] = ['name' => 'veo_zugang', 'label' => 'Zugang ab'];
    $add[] = ['name' => 'veo_laufzeit', 'label' => 'Laufzeit Erbpacht', 'format' => 'number', 'formatSymbol' => 'Jahre'];
    $add[] = ['name' => 'veo_max_personen', 'label' => 'Maximale Personenzahl', 'format' => 'number'];
    $add[] = ['name' => 'veo_nichtraucher', 'label' => 'Nichtraucherobjekt'];
    $add[] = ['name' => 'veo_haustiere', 'label' => 'Haustiere erlaubt'];
    # $add[] = ['name' => 'veo_geschlecht_geschl_attr', 'label' => 'Geschlecht Geschl Attribute'];
    $add[] = ['name' => 'veo_denkmalgeschuetzt', 'label' => 'Denkmalschutzobjekt'];
    # $add[] = ['name' => 'veo_als_ferien', 'label' => 'Als Ferienimmoblie geeignet'];
    # $add[] = ['name' => 'veo_gewerbliche_nutzung', 'label' => 'Gewerbliche Nutzung möglich'];
    # $add[] = ['name' => 'veo_branchen', 'label' => 'Branchen'];
    $add[] = ['name' => 'veo_hochhaus', 'label' => 'Hochhausobjekt'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

function immoAttrListVetBasic(array $tableList) {

    $add = [];

    # $add[] = ['name' => 'vet_objektnr_intern', 'label' => 'Objektnummer intern'];
    $add[] = ['name' => 'vet_objektnr_extern', 'label' => 'Objektnummer'];
    # $add[] = ['name' => 'vet_aktion', 'label' => 'Aktion'];
    # $add[] = ['name' => 'vet_aktiv_von', 'label' => 'Aktiv von'];
    # $add[] = ['name' => 'vet_aktiv_bis', 'label' => 'Aktiv bis'];
    # $add[] = ['name' => 'vet_openimmo_obid', 'label' => 'OpenImmo ID'];
    # $add[] = ['name' => 'vet_kennung_ursprung', 'label' => 'Kennung Ursprung'];

    if (!empty($tableList['vet_stand_vom'])) {
        $add[] = ['name' => 'custom_vet_stand_vom', 'label' => 'Letzte Änderung', 'value' => date("d.m.Y", strtotime($tableList['vet_stand_vom']))];
    }

    # $add[] = ['name' => 'vet_weitergabe_generell', 'label' => 'Weitergabe generell'];
    # $add[] = ['name' => 'vet_weitergabe_positiv', 'label' => 'Weitergabe positiv'];
    # $add[] = ['name' => 'vet_weitergabe_negativ', 'label' => 'Weitergabe negativ'];
    # $add[] = ['name' => 'vet_gruppen_kennung', 'label' => 'Gruppen Kennung'];
    # $add[] = ['name' => 'vet_master', 'label' => 'Master'];
    # $add[] = ['name' => 'vet_master_visible', 'label' => 'Master visible'];
    # $add[] = ['name' => 'vet_sprache', 'label' => 'Sprache'];

    $add = immoFormatAttrList($tableList, $add);
    return immoRenderAttrList($add);

}

#######################################################################

# Alle verfügbaren Objektattribute gelistet
# Rohdaten als Referenz und Basis für eigene Listen
# Unterteilt in Bereiche (Geo, Preise, Flächen usw.)
# Achtung, kann bei 1:1 Nutzung sensible Daten preisgeben!

function immoAttrListAnb(array $tableList) {

    $add = [];
    $prefix = 'anb_';

    $add[] = ['name' => 'anbieternr', 'label' => 'Anbieter Nummer'];
    $add[] = ['name' => 'firma', 'label' => 'Firma'];
    $add[] = ['name' => 'openimmo_anid', 'label' => 'OpenImmo Anbieter ID'];
    $add[] = ['name' => 'lizenzkennung', 'label' => 'Lizenzkennung'];
    $add[] = ['name' => 'impressum', 'label' => 'Impressum'];
    $add[] = ['name' => 'impressum_strukt_firmenname', 'label' => 'Impressum Strukt Firmenname'];
    $add[] = ['name' => 'impressum_strukt_firmenanschrift', 'label' => 'Impressum Strukt Firmenanschrift'];
    $add[] = ['name' => 'impressum_strukt_telefon', 'label' => 'Impressum Strukt Telefon'];
    $add[] = ['name' => 'impressum_strukt_vertretungsberechtigter', 'label' => 'Impressum Strukt Vertretungsberechtigter'];
    $add[] = ['name' => 'impressum_strukt_berufsaufsichtsbehoerde', 'label' => 'Impressum Strukt Berufsaufsichtsbehörde'];
    $add[] = ['name' => 'impressum_strukt_handelsregister', 'label' => 'Impressum Strukt Handelsregister'];
    $add[] = ['name' => 'impressum_strukt_handelsregister_nr', 'label' => 'Impressum Strukt Handelsregister Nummer'];
    $add[] = ['name' => 'impressum_strukt_umsst_id', 'label' => 'Impressum Strukt USt ID'];
    $add[] = ['name' => 'impressum_strukt_steuernummer', 'label' => 'Impressum Strukt Steuernummer'];
    $add[] = ['name' => 'impressum_strukt_weiteres', 'label' => 'Impressum Strukt Weiteres'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListObj(array $tableList) {

    $add = [];
    $prefix = 'obj_';

    $add[] = ['name' => 'nutzungsart_wohnen', 'label' => 'Nutzungsart Wohnen'];
    $add[] = ['name' => 'nutzungsart_gewerbe', 'label' => 'Nutzungsart Gewerbe'];
    $add[] = ['name' => 'nutzungsart_anlage', 'label' => 'Nutzungsart Anlage'];
    $add[] = ['name' => 'nutzungsart_waz', 'label' => 'Nutzungsart WAZ'];
    $add[] = ['name' => 'vermarktungsart_kauf', 'label' => 'Vermarktungsart Kauf'];
    $add[] = ['name' => 'vermarktungsart_miete_pacht', 'label' => 'Vermarktungsart Miete/Pacht'];
    $add[] = ['name' => 'vermarktungsart_erbpacht', 'label' => 'Vermarktungsart Erbpacht'];
    $add[] = ['name' => 'vermarktungsart_leasing', 'label' => 'Vermarktungsart Leasing'];
    $add[] = ['name' => 'objektart', 'label' => 'Objektart'];
    $add[] = ['name' => 'objektart_typ', 'label' => 'Objektart Typ'];
    $add[] = ['name' => 'objektart_objektart_zusatz', 'label' => 'Objektart Objektart Zusatz'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListGeo(array $tableList) {

    $add = [];
    $prefix = 'geo_';

    $add[] = ['name' => 'plz', 'label' => 'PLZ'];
    $add[] = ['name' => 'ort', 'label' => 'Ort'];
    $add[] = ['name' => 'geokoordinaten_breitengrad', 'label' => 'Geokoordinaten Breitengrad'];
    $add[] = ['name' => 'geokoordinaten_laengengrad', 'label' => 'Geokoordinaten Längengrad'];
    $add[] = ['name' => 'strasse', 'label' => 'Straße'];
    $add[] = ['name' => 'hausnummer', 'label' => 'Hausnummer'];
    $add[] = ['name' => 'bundesland', 'label' => 'Bundesland'];
    $add[] = ['name' => 'land_iso_land', 'label' => 'Land ISO Land'];
    $add[] = ['name' => 'gemeindecode', 'label' => 'Gemeindecode'];
    $add[] = ['name' => 'flur', 'label' => 'Flur'];
    $add[] = ['name' => 'flurstueck', 'label' => 'Flurstück'];
    $add[] = ['name' => 'gemarkung', 'label' => 'Gemarkung'];
    $add[] = ['name' => 'etage', 'label' => 'Etage'];
    $add[] = ['name' => 'anzahl_etagen', 'label' => 'Anzahl Etagen'];
    $add[] = ['name' => 'lage_im_bau_links', 'label' => 'Lage im Bau links'];
    $add[] = ['name' => 'lage_im_bau_rechts', 'label' => 'Lage im Bau rechts'];
    $add[] = ['name' => 'lage_im_bau_vorne', 'label' => 'Lage im Bau vorne'];
    $add[] = ['name' => 'lage_im_bau_hinten', 'label' => 'Lage im Bau hinten'];
    $add[] = ['name' => 'wohnungsnr', 'label' => 'Wohnungsnummer'];
    $add[] = ['name' => 'lage_gebiet_gebiete', 'label' => 'Lage Gebiet Gebiete'];
    $add[] = ['name' => 'regionaler_zusatz', 'label' => 'Regionaler Zusatz'];
    $add[] = ['name' => 'karten_makro', 'label' => 'Anzeige Karte Makro'];
    $add[] = ['name' => 'karten_mikro', 'label' => 'Anzeige Karte Mikro'];
    $add[] = ['name' => 'virtuelletour', 'label' => 'Objekt hat virtuelle Tour'];
    $add[] = ['name' => 'luftbildern', 'label' => 'Objekt hat Luftbilder'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListKon(array $tableList) {

    $add = [];
    $prefix = 'kon_';

    $add[] = ['name' => 'vorname', 'label' => 'Vorname'];
    $add[] = ['name' => 'name', 'label' => 'Name'];
    $add[] = ['name' => 'titel', 'label' => 'Titel'];
    $add[] = ['name' => 'anrede', 'label' => 'Anrede'];
    $add[] = ['name' => 'anrede_brief', 'label' => 'Anrede Brief'];
    $add[] = ['name' => 'firma', 'label' => 'Firma'];
    $add[] = ['name' => 'zusatzfeld', 'label' => 'Zusatzfeld'];
    $add[] = ['name' => 'strasse', 'label' => 'Straße'];
    $add[] = ['name' => 'hausnummer', 'label' => 'Hausnummer'];
    $add[] = ['name' => 'plz', 'label' => 'PLZ'];
    $add[] = ['name' => 'ort', 'label' => 'Ort'];
    $add[] = ['name' => 'postfach', 'label' => 'Postfach'];
    $add[] = ['name' => 'postf_plz', 'label' => 'Postfach PLZ'];
    $add[] = ['name' => 'postf_ort', 'label' => 'Postfach Ort'];
    $add[] = ['name' => 'land_iso_land', 'label' => 'Land ISO Land'];
    $add[] = ['name' => 'email_zentrale', 'label' => 'E-Mail Zentrale'];
    $add[] = ['name' => 'email_direkt', 'label' => 'E-Mail direkt'];
    $add[] = ['name' => 'email_privat', 'label' => 'E-Mail privat'];
    $add[] = ['name' => 'email_sonstige', 'label' => 'E-Mail sonstige'];
    $add[] = ['name' => 'email_sonstige_emailart', 'label' => 'E-Mail sonstige Art'];
    $add[] = ['name' => 'email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung'];
    $add[] = ['name' => 'tel_durchw', 'label' => 'Telefon Durchwahl'];
    $add[] = ['name' => 'tel_zentrale', 'label' => 'Telefon Zentrale'];
    $add[] = ['name' => 'tel_handy', 'label' => 'Telefon Handy'];
    $add[] = ['name' => 'tel_fax', 'label' => 'Telefon Fax'];
    $add[] = ['name' => 'tel_privat', 'label' => 'Telefon privat'];
    $add[] = ['name' => 'tel_sonstige', 'label' => 'Telefon sonstige'];
    $add[] = ['name' => 'tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart'];
    $add[] = ['name' => 'tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung'];
    $add[] = ['name' => 'url', 'label' => 'URL'];
    $add[] = ['name' => 'adressfreigabe', 'label' => 'Adressfreigabe'];
    $add[] = ['name' => 'personennummer', 'label' => 'Personennummer'];
    $add[] = ['name' => 'freitextfeld', 'label' => 'Freitextfeld'];
    $add[] = ['name' => 'position', 'label' => 'Position'];
    $add[] = ['name' => 'email_feedback', 'label' => 'E-Mail Feedback'];
    $add[] = ['name' => 'immobilientreuhaenderid', 'label' => 'Immobilientreuhänder ID'];
    $add[] = ['name' => 'referenz_id', 'label' => 'Referenz ID'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListWei(array $tableList) {

    $add = [];
    $prefix = 'wei_';

    $add[] = ['name' => 'vorname', 'label' => 'Vorname'];
    $add[] = ['name' => 'name', 'label' => 'Name'];
    $add[] = ['name' => 'titel', 'label' => 'Titel'];
    $add[] = ['name' => 'anrede', 'label' => 'Anrede'];
    $add[] = ['name' => 'anrede_brief', 'label' => 'Anrede Brief'];
    $add[] = ['name' => 'firma', 'label' => 'Firma'];
    $add[] = ['name' => 'zusatzfeld', 'label' => 'Zusatzfeld'];
    $add[] = ['name' => 'strasse', 'label' => 'Straße'];
    $add[] = ['name' => 'hausnummer', 'label' => 'Hausnummer'];
    $add[] = ['name' => 'plz', 'label' => 'PLZ'];
    $add[] = ['name' => 'ort', 'label' => 'Ort'];
    $add[] = ['name' => 'postfach', 'label' => 'Postfach'];
    $add[] = ['name' => 'postf_plz', 'label' => 'Postfach PLZ'];
    $add[] = ['name' => 'postf_ort', 'label' => 'Postfach Ort'];
    $add[] = ['name' => 'land_iso_land', 'label' => 'Land ISO Land'];
    $add[] = ['name' => 'email_zentrale', 'label' => 'E-Mail Zentrale'];
    $add[] = ['name' => 'email_direkt', 'label' => 'E-Mail direkt'];
    $add[] = ['name' => 'email_privat', 'label' => 'E-Mail privat'];
    $add[] = ['name' => 'email_sonstige', 'label' => 'E-Mail sonstige'];
    $add[] = ['name' => 'email_sonstige_emailart', 'label' => 'E-Mail sonstige Art'];
    $add[] = ['name' => 'email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung'];
    $add[] = ['name' => 'tel_durchw', 'label' => 'Telefon Durchwahl'];
    $add[] = ['name' => 'tel_zentrale', 'label' => 'Telefon Zentrale'];
    $add[] = ['name' => 'tel_handy', 'label' => 'Telefon Handy'];
    $add[] = ['name' => 'tel_fax', 'label' => 'Telefon Fax'];
    $add[] = ['name' => 'tel_privat', 'label' => 'Telefon privat'];
    $add[] = ['name' => 'tel_sonstige', 'label' => 'Telefon sonstige'];
    $add[] = ['name' => 'tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart'];
    $add[] = ['name' => 'tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung'];
    $add[] = ['name' => 'url', 'label' => 'URL'];
    $add[] = ['name' => 'adressfreigabe', 'label' => 'Adressfreigabe'];
    $add[] = ['name' => 'personennummer', 'label' => 'Personennummer'];
    $add[] = ['name' => 'freitextfeld', 'label' => 'Freitextfeld'];
    $add[] = ['name' => 'position', 'label' => 'Position'];
    $add[] = ['name' => 'email_feedback', 'label' => 'E-Mail Feedback'];
    $add[] = ['name' => 'immobilientreuhaenderid', 'label' => 'Immobilientreuhänder ID'];
    $add[] = ['name' => 'referenz_id', 'label' => 'Referenz ID'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListPre(array $tableList) {

    $add = [];
    $prefix = 'pre_';

    $add[] = ['name' => 'kaufpreis', 'label' => 'Kaufpreis'];
    $add[] = ['name' => 'kaufpreis_auf_anfrage', 'label' => 'Kaufpreis auf Anfrage'];
    $add[] = ['name' => 'kaufpreisnetto', 'label' => 'Kaufpreis netto'];
    $add[] = ['name' => 'kaufpreisnetto_kaufpreisust', 'label' => 'Kaufpreis netto USt'];
    $add[] = ['name' => 'kaufpreisbrutto', 'label' => 'Kaufpreis brutto'];
    $add[] = ['name' => 'nettokaltmiete', 'label' => 'Kaltmiete netto'];
    $add[] = ['name' => 'kaltmiete', 'label' => 'Kaltmiete'];
    $add[] = ['name' => 'warmmiete', 'label' => 'Warmmiete'];
    $add[] = ['name' => 'nebenkosten', 'label' => 'Nebenkosten'];
    $add[] = ['name' => 'heizkosten_enthalten', 'label' => 'Heizkosten enthalten'];
    $add[] = ['name' => 'heizkosten', 'label' => 'Heizkosten'];
    $add[] = ['name' => 'zzg_mehrwertsteuer', 'label' => 'Zuzüglich MwSt'];
    $add[] = ['name' => 'mietzuschlaege', 'label' => 'Mietzuschläge'];
    $add[] = ['name' => 'hauptmietzinsnetto', 'label' => 'Hauptmietzins netto'];
    $add[] = ['name' => 'hauptmietzinsnetto_hauptmietzinsust', 'label' => 'Hauptmietzins USt'];
    $add[] = ['name' => 'pauschalmiete', 'label' => 'Pauschalmiete'];
    $add[] = ['name' => 'betriebskostennetto', 'label' => 'Betriebskosten netto'];
    $add[] = ['name' => 'betriebskostennetto_betriebskostenust', 'label' => 'Betriebskosten USt'];
    $add[] = ['name' => 'evbnetto', 'label' => 'EVB netto'];
    $add[] = ['name' => 'evbnetto_evbust', 'label' => 'EVB USt'];
    $add[] = ['name' => 'gesamtmietenetto', 'label' => 'Gesamtmiete netto'];
    $add[] = ['name' => 'gesamtmietenetto_gesamtmieteust', 'label' => 'Gesamtmiete USt'];
    $add[] = ['name' => 'gesamtbelastungbrutto', 'label' => 'Gesamtmiete brutto'];
    $add[] = ['name' => 'gesamtkostenprom2von', 'label' => 'Gesamtkosten pro m² von'];
    $add[] = ['name' => 'gesamtkostenprom2von_gesamtkostenprom2bis', 'label' => 'Gesamtkosten pro m² bis'];
    $add[] = ['name' => 'heizkostennetto', 'label' => 'Heizkosten netto'];
    $add[] = ['name' => 'heizkostennetto_heizkostenust', 'label' => 'Heizkosten USt'];
    $add[] = ['name' => 'monatlichekostennetto', 'label' => 'Monatliche Kosten netto'];
    $add[] = ['name' => 'monatlichekostennetto_monatlichekostenust', 'label' => 'Monatliche Kosten USt'];
    $add[] = ['name' => 'monatlichekostenbrutto', 'label' => 'Monatliche Kosten brutto'];
    $add[] = ['name' => 'nebenkostenprom2von', 'label' => 'Nebenkosten pro m² von'];
    $add[] = ['name' => 'nebenkostenprom2von_nebenkostenprom2bis', 'label' => 'Nebenkosten pro m² bis'];
    $add[] = ['name' => 'ruecklagenetto', 'label' => 'Rücklage netto'];
    $add[] = ['name' => 'ruecklagenetto_ruecklageust', 'label' => 'Rücklage USt'];
    $add[] = ['name' => 'sonstigekostennetto', 'label' => 'Sonstige Kosten netto'];
    $add[] = ['name' => 'ruecklagenetto_sonstigekostenust', 'label' => 'Sonstige Kosten USt'];
    $add[] = ['name' => 'sonstigemietenetto', 'label' => 'Sonstige Miete netto'];
    $add[] = ['name' => 'ruecklagenetto_sonstigemieteust', 'label' => 'Sonstige Miete USt'];
    $add[] = ['name' => 'summemietenetto', 'label' => 'Summe Miete netto'];
    $add[] = ['name' => 'summemietenetto_summemieteust', 'label' => 'Summe Miete USt'];
    $add[] = ['name' => 'nettomieteprom2von', 'label' => 'Nettomiete pro m² von'];
    $add[] = ['name' => 'summemietenetto_nettomieteprom2bis', 'label' => 'Nettomiete pro m² bis'];
    $add[] = ['name' => 'pacht', 'label' => 'Pacht'];
    $add[] = ['name' => 'erbpacht', 'label' => 'Erbpacht'];
    $add[] = ['name' => 'hausgeld', 'label' => 'Hausgeld'];
    $add[] = ['name' => 'abstand', 'label' => 'Abstand'];
    $add[] = ['name' => 'preis_zeitraum_von', 'label' => 'Preis Zeitraum von'];
    $add[] = ['name' => 'preis_zeitraum_bis', 'label' => 'Preis Zeitraum bis'];
    $add[] = ['name' => 'preis_zeiteinheit_zeiteinheit', 'label' => 'Preis Zeiteinheit'];
    $add[] = ['name' => 'mietpreis_pro_qm', 'label' => 'Mietpreis pro m²'];
    $add[] = ['name' => 'kaufpreis_pro_qm', 'label' => 'Kaufpreis pro m²'];
    $add[] = ['name' => 'provisionspflichtig', 'label' => 'Provisionspflichtig'];
    $add[] = ['name' => 'provision_teilen', 'label' => 'Provision teilen'];
    $add[] = ['name' => 'innen_courtage', 'label' => 'Innen Courtage'];
    $add[] = ['name' => 'innen_courtage_mit_mwst', 'label' => 'Innen Courtage mit MwSt'];
    $add[] = ['name' => 'aussen_courtage', 'label' => 'Außen Courtage'];
    $add[] = ['name' => 'aussen_courtage_mit_mwst', 'label' => 'Außen Courtage mit MwSt'];
    $add[] = ['name' => 'courtage_hinweis', 'label' => 'Courtage Hinweis'];
    $add[] = ['name' => 'provisionnetto', 'label' => 'Provision netto'];
    $add[] = ['name' => 'provisionbrutto', 'label' => 'Provision brutto'];
    $add[] = ['name' => 'waehrung_iso_waehrung', 'label' => 'Währung ISO Währung'];
    $add[] = ['name' => 'mwst_satz', 'label' => 'MwSt Satz'];
    $add[] = ['name' => 'mwst_gesamt', 'label' => 'MwSt Gesamt'];
    $add[] = ['name' => 'freitext_preis', 'label' => 'Freitext Preis'];
    $add[] = ['name' => 'x_fache', 'label' => 'X-fache'];
    $add[] = ['name' => 'nettorendite', 'label' => 'Nettorendite'];
    $add[] = ['name' => 'nettorendite_soll', 'label' => 'Nettorendite Soll'];
    $add[] = ['name' => 'nettorendite_ist', 'label' => 'Nettorendite Ist'];
    $add[] = ['name' => 'mieteinnahmen_ist', 'label' => 'Mieteinnahmen Ist'];
    $add[] = ['name' => 'mieteinnahmen_ist_periode', 'label' => 'Mieteinnahmen Ist Periode'];
    $add[] = ['name' => 'mieteinnahmen_soll', 'label' => 'Mieteinnahmen Soll'];
    $add[] = ['name' => 'mieteinnahmen_soll_periode', 'label' => 'Mieteinnahmen Soll Periode'];
    $add[] = ['name' => 'erschliessungskosten', 'label' => 'Erschließungskosten'];
    $add[] = ['name' => 'kaution', 'label' => 'Kaution'];
    $add[] = ['name' => 'kaution_text', 'label' => 'Kaution Text'];
    $add[] = ['name' => 'geschaeftsguthaben', 'label' => 'Geschäftsguthaben'];
    $add[] = ['name' => 'stp_carport_stellplatzmiete', 'label' => 'STP Carport Miete'];
    $add[] = ['name' => 'stp_carport_stellplatzkaufpreis', 'label' => 'STP Carport Kaufpreis'];
    $add[] = ['name' => 'stp_carport_anzahl', 'label' => 'STP Carport Anzahl'];
    $add[] = ['name' => 'stp_duplex_stellplatzmiete', 'label' => 'STP Duplex Miete'];
    $add[] = ['name' => 'stp_duplex_stellplatzkaufpreis', 'label' => 'STP Duplex Kaufpreis'];
    $add[] = ['name' => 'stp_duplex_anzahl', 'label' => 'STP Duplex Anzahl'];
    $add[] = ['name' => 'stp_freiplatz_stellplatzmiete', 'label' => 'STP Freiplatz Miete'];
    $add[] = ['name' => 'stp_freiplatz_stellplatzkaufpreis', 'label' => 'STP Freiplatz Kaufpreis'];
    $add[] = ['name' => 'stp_freiplatz_anzahl', 'label' => 'STP Freiplatz Anzahl'];
    $add[] = ['name' => 'stp_garage_stellplatzmiete', 'label' => 'STP Garage Miete'];
    $add[] = ['name' => 'stp_garage_stellplatzkaufpreis', 'label' => 'STP Garage Kaufpreis'];
    $add[] = ['name' => 'stp_garage_anzahl', 'label' => 'STP Garage Anzahl'];
    $add[] = ['name' => 'stp_parkhaus_stellplatzmiete', 'label' => 'STP Parkhaus Miete'];
    $add[] = ['name' => 'stp_parkhaus_stellplatzkaufpreis', 'label' => 'STP Parkhaus Kaufpreis'];
    $add[] = ['name' => 'stp_parkhaus_anzahl', 'label' => 'STP Parkhaus Anzahl'];
    $add[] = ['name' => 'stp_tiefgarage_stellplatzmiete', 'label' => 'STP Tiefgarage Miete'];
    $add[] = ['name' => 'stp_tiefgarage_stellplatzkaufpreis', 'label' => 'STP Tiefgarage Kaufpreis'];
    $add[] = ['name' => 'stp_tiefgarage_anzahl', 'label' => 'STP Tiefgarage Anzahl'];
    $add[] = ['name' => 'stp_sonstige_stellplatzmiete', 'label' => 'STP Sonstige Miete'];
    $add[] = ['name' => 'stp_sonstige_stellplatzkaufpreis', 'label' => 'STP Sonstige Kaufpreis'];
    $add[] = ['name' => 'stp_sonstige_anzahl', 'label' => 'STP Sonstige Anzahl'];
    $add[] = ['name' => 'stp_sonstige_platzart', 'label' => 'STP Sonstige Platzart'];
    $add[] = ['name' => 'stp_sonstige_bemerkung', 'label' => 'STP Sonstige Bemerkung'];
    $add[] = ['name' => 'richtpreis', 'label' => 'Richtpreis'];
    $add[] = ['name' => 'richtpreisprom2', 'label' => 'Richtpreis pro m²'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListBie(array $tableList) {

    $add = [];
    $prefix = 'bie_';

    $add[] = ['name' => 'beginn_angebotsphase', 'label' => 'Beginn Angebotsphase'];
    $add[] = ['name' => 'besichtigungstermin', 'label' => 'Besichtigungstermin'];
    $add[] = ['name' => 'besichtigungstermin_2', 'label' => 'Besichtigungstermin 2'];
    $add[] = ['name' => 'beginn_bietzeit', 'label' => 'Beginn Bietzeit'];
    $add[] = ['name' => 'ende_bietzeit', 'label' => 'Ende Bietzeit'];
    $add[] = ['name' => 'hoechstgebot_zeigen', 'label' => 'Höchstgebot zeigen'];
    $add[] = ['name' => 'mindestpreis', 'label' => 'Mindestpreis'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListVer(array $tableList) {

    $add = [];
    $prefix = 'ver_';

    $add[] = ['name' => 'zwangsversteigerung', 'label' => 'Zwangsversteigerung'];
    $add[] = ['name' => 'aktenzeichen', 'label' => 'Aktenzeichen'];
    $add[] = ['name' => 'zvtermin', 'label' => 'ZV Termin'];
    $add[] = ['name' => 'zusatztermin', 'label' => 'Zusatztermin'];
    $add[] = ['name' => 'amtsgericht', 'label' => 'Amtsgericht'];
    $add[] = ['name' => 'verkehrswert', 'label' => 'Verkehrswert'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListFla(array $tableList) {

    $add = [];
    $prefix = 'fla_';

    $add[] = ['name' => 'wohnflaeche', 'label' => 'Wohnfläche'];
    $add[] = ['name' => 'nutzflaeche', 'label' => 'Nutzfläche'];
    $add[] = ['name' => 'gesamtflaeche', 'label' => 'Gesamtfläche'];
    $add[] = ['name' => 'ladenflaeche', 'label' => 'Ladenfläche'];
    $add[] = ['name' => 'lagerflaeche', 'label' => 'Lagerfläche'];
    $add[] = ['name' => 'verkaufsflaeche', 'label' => 'Verkaufsfläche'];
    $add[] = ['name' => 'freiflaeche', 'label' => 'Freifläche'];
    $add[] = ['name' => 'bueroflaeche', 'label' => 'Bürofläche'];
    $add[] = ['name' => 'bueroteilflaeche', 'label' => 'Büroteilfläche'];
    $add[] = ['name' => 'fensterfront', 'label' => 'Fensterfront'];
    $add[] = ['name' => 'verwaltungsflaeche', 'label' => 'Verwaltungsfläche'];
    $add[] = ['name' => 'gastroflaeche', 'label' => 'Gastrofläche'];
    $add[] = ['name' => 'grz', 'label' => 'GRZ'];
    $add[] = ['name' => 'gfz', 'label' => 'GFZ'];
    $add[] = ['name' => 'bmz', 'label' => 'BMZ'];
    $add[] = ['name' => 'bgf', 'label' => 'BGF'];
    $add[] = ['name' => 'grundstuecksflaeche', 'label' => 'Grundstücksfläche'];
    $add[] = ['name' => 'sonstflaeche', 'label' => 'Sonstige Fläche'];
    $add[] = ['name' => 'anzahl_zimmer', 'label' => 'Anzahl Zimmer'];
    $add[] = ['name' => 'anzahl_schlafzimmer', 'label' => 'Anzahl Schlafzimmer'];
    $add[] = ['name' => 'anzahl_badezimmer', 'label' => 'Anzahl Badezimmer'];
    $add[] = ['name' => 'anzahl_sep_wc', 'label' => 'Anzahl separate WCs'];
    $add[] = ['name' => 'anzahl_balkone', 'label' => 'Anzahl Balkone'];
    $add[] = ['name' => 'anzahl_terrassen', 'label' => 'Anzahl Terrassen'];
    $add[] = ['name' => 'anzahl_logia', 'label' => 'Anzahl Logia'];
    $add[] = ['name' => 'balkon_terrasse_flaeche', 'label' => 'Fläche aller Balkon/Terrasse'];
    $add[] = ['name' => 'anzahl_wohn_schlafzimmer', 'label' => 'Anzahl Wohn-Schlafzimmer'];
    $add[] = ['name' => 'gartenflaeche', 'label' => 'Gartenfläche'];
    $add[] = ['name' => 'kellerflaeche', 'label' => 'Kellerfläche'];
    $add[] = ['name' => 'fensterfront_qm', 'label' => 'Fensterfront'];
    $add[] = ['name' => 'grundstuecksfront', 'label' => 'Grundstücksfront in m'];
    $add[] = ['name' => 'dachbodenflaeche', 'label' => 'Dachbodenfläche'];
    $add[] = ['name' => 'teilbar_ab', 'label' => 'Teilbar ab'];
    $add[] = ['name' => 'beheizbare_flaeche', 'label' => 'Beheizbare Fläche'];
    $add[] = ['name' => 'anzahl_stellplaetze', 'label' => 'Anzahl Stellplätze'];
    $add[] = ['name' => 'plaetze_gastraum', 'label' => 'Plätze Gastraum'];
    $add[] = ['name' => 'anzahl_betten', 'label' => 'Anzahl Betten'];
    $add[] = ['name' => 'anzahl_tagungsraeume', 'label' => 'Anzahl Tagungsräume'];
    $add[] = ['name' => 'vermietbare_flaeche', 'label' => 'Vermietbare Fläche'];
    $add[] = ['name' => 'anzahl_wohneinheiten', 'label' => 'Anzahl Wohneinheiten'];
    $add[] = ['name' => 'anzahl_gewerbeeinheiten', 'label' => 'Anzahl Gewerbeeinheiten'];
    $add[] = ['name' => 'einliegerwohnung', 'label' => 'Einliegerwohnung'];
    $add[] = ['name' => 'kubatur', 'label' => 'Kubatur'];
    $add[] = ['name' => 'ausnuetzungsziffer', 'label' => 'Ausnützungsziffer'];
    $add[] = ['name' => 'flaechevon', 'label' => 'Fläche von'];
    $add[] = ['name' => 'flaechebis', 'label' => 'Fläche bis'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListAus(array $tableList) {

    $add = [];
    $prefix = 'aus_';

    $add[] = ['name' => 'ausstatt_kategorie', 'label' => 'Ausstattung Kategorie'];
    $add[] = ['name' => 'wg_geeignet', 'label' => 'WG geeignet'];
    $add[] = ['name' => 'raeume_veraenderbar', 'label' => 'Räume veränderbar'];
    $add[] = ['name' => 'bad_dusche', 'label' => 'Bad Dusche'];
    $add[] = ['name' => 'bad_wanne', 'label' => 'Bad Wanne'];
    $add[] = ['name' => 'bad_fenster', 'label' => 'Bad Fenster'];
    $add[] = ['name' => 'bad_bidet', 'label' => 'Bad Bidet'];
    $add[] = ['name' => 'bad_pissoir', 'label' => 'Bad Pissoir'];
    $add[] = ['name' => 'kueche_ebk', 'label' => 'Küche EBK'];
    $add[] = ['name' => 'kueche_offen', 'label' => 'Küche offen'];
    $add[] = ['name' => 'kueche_pantry', 'label' => 'Küche Pantry'];
    $add[] = ['name' => 'boden_fliesen', 'label' => 'Boden Fliesen'];
    $add[] = ['name' => 'boden_stein', 'label' => 'Boden Stein'];
    $add[] = ['name' => 'boden_teppich', 'label' => 'Boden Teppich'];
    $add[] = ['name' => 'boden_parkett', 'label' => 'Boden Parkett'];
    $add[] = ['name' => 'boden_fertigparkett', 'label' => 'Boden Fertigparkett'];
    $add[] = ['name' => 'boden_laminat', 'label' => 'Boden Laminat'];
    $add[] = ['name' => 'boden_dielen', 'label' => 'Boden Dielen'];
    $add[] = ['name' => 'boden_kunststoff', 'label' => 'Boden Kunststoff'];
    $add[] = ['name' => 'boden_estrich', 'label' => 'Boden Estrich'];
    $add[] = ['name' => 'boden_doppelboden', 'label' => 'Boden Doppelboden'];
    $add[] = ['name' => 'boden_linoleum', 'label' => 'Boden Linoleum'];
    $add[] = ['name' => 'boden_marmor', 'label' => 'Boden Marmor'];
    $add[] = ['name' => 'boden_terrakotta', 'label' => 'Boden Terrakotta'];
    $add[] = ['name' => 'boden_granit', 'label' => 'Boden Granit'];
    $add[] = ['name' => 'kamin', 'label' => 'Kamin'];
    $add[] = ['name' => 'heizungsart_ofen', 'label' => 'Heizungsart Ofen'];
    $add[] = ['name' => 'heizungsart_etage', 'label' => 'Heizungsart Etage'];
    $add[] = ['name' => 'heizungsart_zentral', 'label' => 'Heizungsart zentral'];
    $add[] = ['name' => 'heizungsart_fern', 'label' => 'Heizungsart fern'];
    $add[] = ['name' => 'heizungsart_fussboden', 'label' => 'Heizungsart Fußboden'];
    $add[] = ['name' => 'befeuerung_oel', 'label' => 'Befeuerung Öl'];
    $add[] = ['name' => 'befeuerung_gas', 'label' => 'Befeuerung Gas'];
    $add[] = ['name' => 'befeuerung_elektro', 'label' => 'Befeuerung Elektro'];
    $add[] = ['name' => 'befeuerung_alternativ', 'label' => 'Befeuerung alternativ'];
    $add[] = ['name' => 'befeuerung_solar', 'label' => 'Befeuerung Solar'];
    $add[] = ['name' => 'befeuerung_erdwaerme', 'label' => 'Befeuerung Erdwärme'];
    $add[] = ['name' => 'befeuerung_luftwp', 'label' => 'Befeuerung Luft WP'];
    $add[] = ['name' => 'befeuerung_fern', 'label' => 'Befeuerung fern'];
    $add[] = ['name' => 'befeuerung_block', 'label' => 'Befeuerung Block'];
    $add[] = ['name' => 'befeuerung_wasser_elektro', 'label' => 'Befeuerung Wasser Elektro'];
    $add[] = ['name' => 'befeuerung_pellet', 'label' => 'Befeuerung Pellet'];
    $add[] = ['name' => 'befeuerung_kohle', 'label' => 'Befeuerung Kohle'];
    $add[] = ['name' => 'befeuerung_holz', 'label' => 'Befeuerung Holz'];
    $add[] = ['name' => 'befeuerung_fluessiggas', 'label' => 'Befeuerung Flüssiggas'];
    $add[] = ['name' => 'klimatisiert', 'label' => 'Klimatisiert'];
    $add[] = ['name' => 'fahrstuhl_personen', 'label' => 'Fahrstuhl Personen'];
    $add[] = ['name' => 'fahrstuhl_lasten', 'label' => 'Fahrstuhl Lasten'];
    $add[] = ['name' => 'stellplatzart_garage', 'label' => 'Stellplatzart Garage'];
    $add[] = ['name' => 'stellplatzart_tiefgarage', 'label' => 'Stellplatzart Tiefgarage'];
    $add[] = ['name' => 'stellplatzart_carport', 'label' => 'Stellplatzart Carport'];
    $add[] = ['name' => 'stellplatzart_freiplatz', 'label' => 'Stellplatzart Freiplatz'];
    $add[] = ['name' => 'stellplatzart_parkhaus', 'label' => 'Stellplatzart Parkhaus'];
    $add[] = ['name' => 'stellplatzart_duplex', 'label' => 'Stellplatzart Duplex'];
    $add[] = ['name' => 'gartennutzung', 'label' => 'Gartennutzung'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_nord', 'label' => 'Ausrichtung Balkon Terrasse Nord'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_ost', 'label' => 'Ausrichtung Balkon Terrasse Ost'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_sued', 'label' => 'Ausrichtung Balkon Terrasse Süd'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_west', 'label' => 'Ausrichtung Balkon Terrasse West'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_nordost', 'label' => 'Ausrichtung Balkon Terrasse Nordost'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_nordwest', 'label' => 'Ausrichtung Balkon Terrasse Nordest'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_suedost', 'label' => 'Ausrichtung Balkon Terrasse Südost'];
    $add[] = ['name' => 'ausricht_balkon_terrasse_suedwest', 'label' => 'Ausrichtung Balkon Terrasse Südwest'];
    $add[] = ['name' => 'moebliert_moeb', 'label' => 'Möbliert Möb'];
    $add[] = ['name' => 'rollstuhlgerecht', 'label' => 'Rollstuhlgerecht'];
    $add[] = ['name' => 'kabel_sat_tv', 'label' => 'Kabel Sat TV'];
    $add[] = ['name' => 'dvbt', 'label' => 'DVBT'];
    $add[] = ['name' => 'barrierefrei', 'label' => 'Barrierefrei'];
    $add[] = ['name' => 'sauna', 'label' => 'Sauna'];
    $add[] = ['name' => 'swimmingpool', 'label' => 'Swimmingpool'];
    $add[] = ['name' => 'wasch_trockenraum', 'label' => 'Wasch-/Trockenraum'];
    $add[] = ['name' => 'wintergarten', 'label' => 'Wintergarten'];
    $add[] = ['name' => 'dv_verkabelung', 'label' => 'DV-Verkabelung'];
    $add[] = ['name' => 'rampe', 'label' => 'Rampe'];
    $add[] = ['name' => 'hebebuehne', 'label' => 'Hebebühne'];
    $add[] = ['name' => 'kran', 'label' => 'Kran'];
    $add[] = ['name' => 'gastterrasse', 'label' => 'Gastterrasse'];
    $add[] = ['name' => 'stromanschlusswert', 'label' => 'Stromanschlusswert'];
    $add[] = ['name' => 'kantine_cafeteria', 'label' => 'Kantine Cafeteria'];
    $add[] = ['name' => 'teekueche', 'label' => 'Teeküche'];
    $add[] = ['name' => 'hallenhoehe', 'label' => 'Hallenhöhe'];
    $add[] = ['name' => 'angeschl_gastronomie_hotelrestaurant', 'label' => 'Angeschlossene Gastronomie Hotelrestaurant'];
    $add[] = ['name' => 'angeschl_gastronomie_bar', 'label' => 'Angeschlossene Gastronomie Bar'];
    $add[] = ['name' => 'brauereibindung', 'label' => 'Brauereibindung'];
    $add[] = ['name' => 'sporteinrichtungen', 'label' => 'Sporteinrichtungen'];
    $add[] = ['name' => 'wellnessbereich', 'label' => 'Wellnessbereich'];
    $add[] = ['name' => 'serviceleistungen_betreutes_wohnen', 'label' => 'Serviceleistungen Betreutes Wohnen'];
    $add[] = ['name' => 'serviceleistungen_catering', 'label' => 'Serviceleistungen Catering'];
    $add[] = ['name' => 'serviceleistungen_reinigung', 'label' => 'Serviceleistungen Reinigung'];
    $add[] = ['name' => 'serviceleistungen_einkauf', 'label' => 'Serviceleistungen Einkauf'];
    $add[] = ['name' => 'serviceleistungen_wachdienst', 'label' => 'Serviceleistungen Wachdienst'];
    $add[] = ['name' => 'telefon_ferienimmobilie', 'label' => 'Telefon Ferienimmobilie'];
    $add[] = ['name' => 'breitband_zugang_art', 'label' => 'Breitband Zugang Art'];
    $add[] = ['name' => 'breitband_zugang_speed', 'label' => 'Breitband Zugang Speed'];
    $add[] = ['name' => 'umts_empfang', 'label' => 'UMTS Empfang'];
    $add[] = ['name' => 'sicherheitstechnik_alarmanlage', 'label' => 'Sicherheitstechnik Alarmanlage'];
    $add[] = ['name' => 'sicherheitstechnik_kamera', 'label' => 'Sicherheitstechnik Kamera'];
    $add[] = ['name' => 'sicherheitstechnik_polizeiruf', 'label' => 'Sicherheitstechnik Polizeiruf'];
    $add[] = ['name' => 'unterkellert_keller', 'label' => 'Unterkellert Keller'];
    $add[] = ['name' => 'abstellraum', 'label' => 'Abstellraum'];
    $add[] = ['name' => 'fahrradraum', 'label' => 'Fahrradraum'];
    $add[] = ['name' => 'rolladen', 'label' => 'Rolladen'];
    $add[] = ['name' => 'dachform_krueppelwalmdach', 'label' => 'Dachform Krüppelwalmdach'];
    $add[] = ['name' => 'dachform_mansarddach', 'label' => 'Dachform Mansarddach'];
    $add[] = ['name' => 'dachform_pultdach', 'label' => 'Dachform Pultdach'];
    $add[] = ['name' => 'dachform_satteldach', 'label' => 'Dachform Satteldach'];
    $add[] = ['name' => 'dachform_walmdach', 'label' => 'Dachform Walmdach'];
    $add[] = ['name' => 'dachform_flachdach', 'label' => 'Dachform Flachdach'];
    $add[] = ['name' => 'dachform_pyramidendach', 'label' => 'Dachform Pyramidendach'];
    $add[] = ['name' => 'bauweise_massiv', 'label' => 'Bauweise massiv'];
    $add[] = ['name' => 'bauweise_fertigteile', 'label' => 'Bauweise Fertigteile'];
    $add[] = ['name' => 'bauweise_holz', 'label' => 'Bauweise Holz'];
    $add[] = ['name' => 'ausbaustufe_bausatzhaus', 'label' => 'Ausbaustufe Bausatzhaus'];
    $add[] = ['name' => 'ausbaustufe_ausbauhaus', 'label' => 'Ausbaustufe Ausbauhaus'];
    $add[] = ['name' => 'ausbaustufe_schluesselfertigmitkeller', 'label' => 'Ausbaustufe Schlüsselfertig mit Keller'];
    $add[] = ['name' => 'ausbaustufe_schluesselfertigohnebodenplatte', 'label' => 'Ausbaustufe Schlüsselfertig ohne Bodenplatte'];
    $add[] = ['name' => 'ausbaustufe_schluesselfertigmitbodenplatte', 'label' => 'Ausbaustufe Schlüsselfertig mit Bodenplatte'];
    $add[] = ['name' => 'energietyp_passivhaus', 'label' => 'Energietyp Passivhaus'];
    $add[] = ['name' => 'energietyp_niedrigenergie', 'label' => 'Energietyp Niedrigenergie'];
    $add[] = ['name' => 'energietyp_neubaustandard', 'label' => 'Energietyp Neubaustandard'];
    $add[] = ['name' => 'energietyp_kfw40', 'label' => 'Energietyp Kfw 40'];
    $add[] = ['name' => 'energietyp_kfw60', 'label' => 'Energietyp Kfw 60'];
    $add[] = ['name' => 'energietyp_kfw55', 'label' => 'Energietyp Kfw 55'];
    $add[] = ['name' => 'energietyp_kfw70', 'label' => 'Energietyp Kfw 70'];
    $add[] = ['name' => 'energietyp_minergiebauweise', 'label' => 'Energietyp Minergiebauweise'];
    $add[] = ['name' => 'energietyp_minergie_zertifiziert', 'label' => 'Energietyp Minergie zertifiziert'];
    $add[] = ['name' => 'bibliothek', 'label' => 'Bibliothek'];
    $add[] = ['name' => 'dachboden', 'label' => 'Dachboden'];
    $add[] = ['name' => 'gaestewc', 'label' => 'Gäste WC'];
    $add[] = ['name' => 'kabelkanaele', 'label' => 'Kabelkanäle'];
    $add[] = ['name' => 'seniorengerecht', 'label' => 'Seniorengerecht'];



    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListZus(array $tableList) {

    $add = [];
    $prefix = 'zus_';

    $add[] = ['name' => 'baujahr', 'label' => 'Baujahr'];
    $add[] = ['name' => 'letztemodernisierung', 'label' => 'Letzte Modernisierung'];
    $add[] = ['name' => 'zustand_zustand_art', 'label' => 'Zustand Zustand Art'];
    $add[] = ['name' => 'alter_alter_attr', 'label' => 'Alter Alter Attr'];
    $add[] = ['name' => 'bebaubar_nach_bebaubar_attr', 'label' => 'Bebaubar nach Attribut'];
    $add[] = ['name' => 'erschliessung_erschl_attr', 'label' => 'Erschließung Attribut'];
    $add[] = ['name' => 'erschliessung_umfang_erschl_attr', 'label' => 'Erschließung Umfang Attribut'];
    $add[] = ['name' => 'bauzone', 'label' => 'Bauzone'];
    $add[] = ['name' => 'altlasten', 'label' => 'Altlasten'];
    $add[] = ['name' => 'energiepass_epart', 'label' => 'Energiepass epart'];
    $add[] = ['name' => 'energiepass_gueltig_bis', 'label' => 'Energiepass gültig bis'];
    $add[] = ['name' => 'energiepass_energieverbrauchkennwert', 'label' => 'Energiepass Energieverbrauchkennwert'];
    $add[] = ['name' => 'energiepass_mitwarmwasser', 'label' => 'Energiepass mit Warmwasser'];
    $add[] = ['name' => 'energiepass_endenergiebedarf', 'label' => 'Energiepass Endenergiebedarf'];
    $add[] = ['name' => 'energiepass_primaerenergietraeger', 'label' => 'Energiepass Primärenergieträger'];
    $add[] = ['name' => 'energiepass_stromwert', 'label' => 'Energiepass Stromwert'];
    $add[] = ['name' => 'energiepass_waermewert', 'label' => 'Energiepass Wärmewert'];
    $add[] = ['name' => 'energiepass_wertklasse', 'label' => 'Energiepass Wertklasse'];
    $add[] = ['name' => 'energiepass_baujahr', 'label' => 'Energiepass Baujahr'];
    $add[] = ['name' => 'energiepass_ausstelldatum', 'label' => 'Energiepass Ausstelldatum'];
    $add[] = ['name' => 'energiepass_jahrgang', 'label' => 'Energiepass Jahrgang'];
    $add[] = ['name' => 'energiepass_gebaeudeart', 'label' => 'Energiepass Gebäudeart'];
    $add[] = ['name' => 'energiepass_epasstext', 'label' => 'Energiepass Epass Text'];
    $add[] = ['name' => 'energiepass_geg2018', 'label' => 'Energiepass GEG 2018'];
    $add[] = ['name' => 'hwbwert', 'label' => 'Heizwärmebedarf Wert'];
    $add[] = ['name' => 'hwbklasse', 'label' => 'Heizwärmebedarf Klasse'];
    $add[] = ['name' => 'fgeewert', 'label' => 'Gesamtenergieeffizienz-Faktor Wert'];
    $add[] = ['name' => 'fgeeklasse', 'label' => 'Gesamtenergieeffizienz-Faktor Klasse'];
    $add[] = ['name' => 'verkaufstatus_stand', 'label' => 'Verkaufstatus Stand'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListBew(array $tableList) {

    $add = [];
    $prefix = 'bew_';

    $add[] = ['name' => 'feld_name', 'label' => 'Name'];
    $add[] = ['name' => 'feld_wert', 'label' => 'Wert'];
    $add[] = ['name' => 'feld_typ', 'label' => 'Typ'];
    $add[] = ['name' => 'feld_modus', 'label' => 'Modus'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListInf(array $tableList) {

    $add = [];
    $prefix = 'inf_';

    $add[] = ['name' => 'zulieferung', 'label' => 'Zulieferung'];
    $add[] = ['name' => 'ausblick_blick', 'label' => 'Ausblick Blick'];
    $add[] = ['name' => 'distanzen_distanz_zu_autobahn', 'label' => 'Distanz zur Autobahn'];
    $add[] = ['name' => 'distanzen_distanz_zu_bus', 'label' => 'Distanz zum Bus'];
    $add[] = ['name' => 'distanzen_distanz_zu_fernbahnhof', 'label' => 'Distanz zum Fernbahnhof'];
    $add[] = ['name' => 'distanzen_distanz_zu_flughafen', 'label' => 'Distanz zum Flughafen'];
    $add[] = ['name' => 'distanzen_distanz_zu_gesamtschule', 'label' => 'Distanz zur Gesamtschule'];
    $add[] = ['name' => 'distanzen_distanz_zu_grundschule', 'label' => 'Distanz zur Grundschule'];
    $add[] = ['name' => 'distanzen_distanz_zu_gymnasium', 'label' => 'Distanz zum Gymnasium'];
    $add[] = ['name' => 'distanzen_distanz_zu_hauptschule', 'label' => 'Distanz zur Hauptschule'];
    $add[] = ['name' => 'distanzen_distanz_zu_kindergaerten', 'label' => 'Distanz zur Kindertagesstätte'];
    $add[] = ['name' => 'distanzen_distanz_zu_realschule', 'label' => 'Distanz zur Realschule'];
    $add[] = ['name' => 'distanzen_distanz_zu_us_bahn', 'label' => 'Distanz zum Nahbahnhof (ÖPNV)'];
    $add[] = ['name' => 'distanzen_distanz_zu_zentrum', 'label' => 'Distanz zum Zentrum'];
    $add[] = ['name' => 'distanzen_distanz_zu_einkaufsmoeglichkeiten', 'label' => 'Distanz zu Einkaufsmöglichkeiten'];
    $add[] = ['name' => 'distanzen_distanz_zu_gaststaetten', 'label' => 'Distanz zu Gaststätten'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_meer', 'label' => 'Distanz zum Meer'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_naherholung', 'label' => 'Distanz zur Naherholung'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_see', 'label' => 'Distanz zum See'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_skigebiet', 'label' => 'Distanz zum Skigebiet'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_sportanlagen', 'label' => 'Distanz zu Sportanlagen'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_strand', 'label' => 'Distanz zum Strand'];
    $add[] = ['name' => 'distanzen_sport_distanz_zu_sport_wandergebiete', 'label' => 'Distanz zum Wandergebiet'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListFre(array $tableList) {

    $add = [];
    $prefix = 'fre_';

    $add[] = ['name' => 'objekttitel', 'label' => 'Objekttitel'];
    $add[] = ['name' => 'dreizeiler', 'label' => 'Dreizeiler'];
    $add[] = ['name' => 'lage', 'label' => 'Lage'];
    $add[] = ['name' => 'ausstatt_beschr', 'label' => 'Ausstatt Beschreibung'];
    $add[] = ['name' => 'objektbeschreibung', 'label' => 'Objektbeschreibung'];
    $add[] = ['name' => 'sonstige_angaben', 'label' => 'Sonstige Angaben'];
    $add[] = ['name' => 'objekt_text', 'label' => 'Objekt Text'];
    $add[] = ['name' => 'objekt_text_lang', 'label' => 'Objekt Text Lang'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListVeo(array $tableList) {

    $add = [];
    $prefix = 'veo_';

    $add[] = ['name' => 'objektadresse_freigeben', 'label' => 'Objektadresse freigeben'];
    $add[] = ['name' => 'verfuegbar_ab', 'label' => 'Verfügbar ab'];
    $add[] = ['name' => 'abdatum', 'label' => 'Ab Datum'];
    $add[] = ['name' => 'bisdatum', 'label' => 'Bis Datum'];
    $add[] = ['name' => 'min_mietdauer_min_dauer', 'label' => 'Min Mietdauer min Dauer'];
    $add[] = ['name' => 'max_mietdauer_max_dauer', 'label' => 'Max Mietdauer max Dauer'];
    $add[] = ['name' => 'versteigerungstermin', 'label' => 'Versteigerungstermin'];
    $add[] = ['name' => 'wbs_sozialwohnung', 'label' => 'WBS Sozialwohnung'];
    $add[] = ['name' => 'vermietet', 'label' => 'Vermietet'];
    $add[] = ['name' => 'gruppennummer', 'label' => 'Gruppennummer'];
    $add[] = ['name' => 'zugang', 'label' => 'Zugang'];
    $add[] = ['name' => 'laufzeit', 'label' => 'Laufzeit'];
    $add[] = ['name' => 'max_personen', 'label' => 'Max Personen'];
    $add[] = ['name' => 'nichtraucher', 'label' => 'Nichtraucher'];
    $add[] = ['name' => 'haustiere', 'label' => 'Haustiere'];
    $add[] = ['name' => 'geschlecht_geschl_attr', 'label' => 'Geschlecht Geschl Attribute'];
    $add[] = ['name' => 'denkmalgeschuetzt', 'label' => 'Denkmalgeschuetzt'];
    $add[] = ['name' => 'als_ferien', 'label' => 'Als Ferien'];
    $add[] = ['name' => 'gewerbliche_nutzung', 'label' => 'Gewerbliche Nutzung'];
    $add[] = ['name' => 'branchen', 'label' => 'Branchen'];
    $add[] = ['name' => 'hochhaus', 'label' => 'Hochhaus'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

function immoAttrListVet(array $tableList) {

    $add = [];
    $prefix = 'vet_';

    $add[] = ['name' => 'objektnr_intern', 'label' => 'Objektnummer intern'];
    $add[] = ['name' => 'objektnr_extern', 'label' => 'Objektnummer extern'];
    $add[] = ['name' => 'aktion', 'label' => 'Aktion'];
    $add[] = ['name' => 'aktiv_von', 'label' => 'Aktiv von'];
    $add[] = ['name' => 'aktiv_bis', 'label' => 'Aktiv bis'];
    $add[] = ['name' => 'openimmo_obid', 'label' => 'OpenImmo ID'];
    $add[] = ['name' => 'kennung_ursprung', 'label' => 'Kennung Ursprung'];
    $add[] = ['name' => 'stand_vom', 'label' => 'Stand vom'];
    $add[] = ['name' => 'weitergabe_generell', 'label' => 'Weitergabe generell'];
    $add[] = ['name' => 'weitergabe_positiv', 'label' => 'Weitergabe positiv'];
    $add[] = ['name' => 'weitergabe_negativ', 'label' => 'Weitergabe negativ'];
    $add[] = ['name' => 'gruppen_kennung', 'label' => 'Gruppen Kennung'];
    $add[] = ['name' => 'master', 'label' => 'Master'];
    $add[] = ['name' => 'master_visible', 'label' => 'Master visible'];
    $add[] = ['name' => 'sprache', 'label' => 'Sprache'];

    $add = immoFormatAttrList($tableList, $add, $prefix);
    return immoRenderAttrList($add);

}

# ---------------------------------------------------------------------

?>
